import os

from plan.common.utils import startrek
from plan.common.utils.hiderefer import get_hidereferred
from plan.contacts import regexes
from plan.exceptions import ContactError, QueueDoesNotExist

# TODO: Вынести в settings
ML_URL = 'http://ml.yandex-team.ru/lists/'
WIKI_URL = 'https://wiki.yandex-team.ru/'
METRIKA_URL = 'http://metrika.yandex.ru/stat/dashboard/?counter_id='
AT_URL = 'http://clubs.at.yandex-team.ru/'
ST_URL = 'https://st.yandex-team.ru/'


class UrlHelperRegistry(dict):
    def register(self, contact_type):
        def decorator(obj):
            self[contact_type] = obj
            return obj

        return decorator


registry = UrlHelperRegistry()


class ContactUrlHelper(object):
    regexes = [regexes.URL_RE]
    url = ''
    normalize_key = None

    def validate(self, content, *args, **kwargs):
        for regex in self.regexes:
            match = regex.match(content)
            if match:
                return
        else:
            raise ContactError

    def normalize(self, content, *args, **kwargs):
        """
        Превращает то, что пользователь ввел в форме в то,
        что мы будем хранить в базе
        """
        for regex in self.regexes:
            match = regex.search(content)
            if match:
                if self.normalize_key is not None:
                    parsed = match.groupdict()
                    return parsed.get(self.normalize_key, content)
                else:
                    return content
        else:
            raise ContactError

    def denormalize(self, content, *args, **kwargs):
        """
        Превращает то, что мы храним в базе в то,
        что нужно показать пользователю
        """
        url = self.url + content
        return get_hidereferred(url)


@registry.register('WIKI')
class WikiUrlHelper(ContactUrlHelper):
    regexes = [regexes.WIKI_RE, regexes.WIKI_PATH_RE]
    url = WIKI_URL
    normalize_key = 'wiki_path'


@registry.register('MAILLIST')
class MlUrlHelper(ContactUrlHelper):
    regexes = [regexes.ML_EMAIL_RE, regexes.ML_RE]
    url = ML_URL
    normalize_key = 'ml'


@registry.register('METRIKA_ID')
class MetrikaUrlHelper(ContactUrlHelper):
    regexes = [regexes.METRIKA_RE]
    url = METRIKA_URL
    normalize_key = 'metrika_id'


@registry.register('AT_CLUB')
class AtUrlHelper(ContactUrlHelper):
    regexes = [regexes.AT_CLUB_RE, regexes.AT_ONLY_CLUB_RE]
    url = AT_URL
    normalize_key = 'at_club'


@registry.register('STARTREK')
class TrackerUrlHelper(ContactUrlHelper):
    regexes = [regexes.QUEUE_RE, regexes.ST_RE]
    normalize_key = 'queue'

    def normalize(self, content, *args, **kwargs):
        queue = super(TrackerUrlHelper, self).normalize(content)
        tracker_type = startrek.get_tracker_type_by_queue(queue)

        if tracker_type != 'startrek':
            raise QueueDoesNotExist

        return queue.upper()

    def denormalize(self, content, *args, **kwargs):
        return get_hidereferred(os.path.join(ST_URL, content))


@registry.register('ST_FILTER')
class STFilterUrlHelper(ContactUrlHelper):
    regexes = [regexes.FILTER_RE, regexes.FILTER_QUERY_RE]


def get_url_helper(contact_type):
    return registry.get(contact_type, ContactUrlHelper)()
