import logging


logger = logging.getLogger(__name__)


def check_model_denormalized_fields(model_path, fields=None, fix=True):
    """
    По умолчанию проверяем все денормализованные поля для модели. Если передано
    fields, то только их. В случае ошибки пишем ворнинг в лог. Если
    fix == True, исправляем ошибку.
    """
    from django.apps import apps
    model_cls = apps.get_model(model_path)
    all_denorm_fields = set(model_cls.DENORMALIZED_FIELDS)

    if fields is None:
        fields = all_denorm_fields
    else:
        fields = set(fields) & all_denorm_fields

    for obj in model_cls.objects.all():
        try:
            check_obj_with_denormalized_fields(obj, fields, fix)
        except Exception:
            logger.exception('Exception during denormalization check')


def check_obj_with_denormalized_fields(obj, fields, fix):
    # собираем для каждого поля текущее и корректное значение
    fields_values = {}
    for field in fields:
        current_value, correct_value = check_denormalized_field(obj, field)
        fields_values[field] = (current_value, correct_value)

    incosistent_fields = {
        field: (current, correct)
        for field, (current, correct) in fields_values.items()
        if current != correct
    }

    # обрабатываем собранное
    for field, values_pair in incosistent_fields.items():
        current_value, correct_value = values_pair
        if current_value != correct_value:
            log_denormalization_warning(
                obj=obj,
                field=field,
                values=(current_value, correct_value)
            )

    if fix and incosistent_fields:
        for field, values_pair in incosistent_fields.items():
            current_value, correct_value = values_pair
            setattr(obj, field, correct_value)
        obj.save(update_fields=incosistent_fields.keys())


def check_denormalized_field(obj, field):
    current_value = getattr(obj, field)
    correct_value = obj.DENORMALIZED_FIELDS[field]['calculator'](obj)

    return current_value, correct_value


def log_denormalization_warning(obj, field, values):
    msg = (
        'Denormalization inconsistency: %(model)s:%(pk)s. '
        '%(field)s: %(current)s, correct value: %(correct)s'
    )
    model_path = '.'.join([obj._meta.app_label, obj._meta.model_name])
    current_value, correct_value = values
    logger.warning(
        msg,
        {
            'model': model_path,
            'pk': obj.pk,
            'field': field,
            'current': current_value,
            'correct': correct_value,
        }
    )
