import logging

from django.db.models import ObjectDoesNotExist

from plan.celery_app import app
from plan.common.utils.tasks import lock_task
from plan.denormalization.check import check_denormalized_field, check_model_denormalized_fields
from plan.denormalization.settings import MODELS_WITH_DENORM_FIELDS
from plan.denormalization.utils import get_model_instance

log = logging.getLogger(__name__)


@app.task
def update_denormalized_field(model_path, pk, denorm_field):
    """
    Общая функция для обновления любого денормализованного поля. В модели
    `model_path` должна содержаться информация, из которой можно понять как
    обновлять поле `denorm_field` у сущности с id `pk`.

    @:param model_path str, например 'services.Service'.

    Параметры из простых типов, потому что эта функция часто будет использована
    как celery-таск и не хочется сериализовать модели, а переполучать ее из
    базы все равно придется, когда задача начнет выполнятся.
    """
    try:
        obj = get_model_instance(model_path, pk)

    except ObjectDoesNotExist as e:
        log.warning('No object %s-%s: %s', model_path, pk, e)
        return False

    current_value, correct_value = check_denormalized_field(obj, denorm_field)

    fix_needed = current_value != correct_value
    if fix_needed:
        setattr(obj, denorm_field, correct_value)
        obj.save(update_fields=[denorm_field])
    return fix_needed


@lock_task
def check_denormalized(model_path=None, fix=True):
    """
    Проверить, что поля `denorm_fields` модели заданной с помощью `model_path`
    содержит корректные данные, если нет, написать warning в лог, и, в случае
    если задан, `fix`=True, исправить.
    По умолчанию берутся все поля указанной модели или все поля всех моделей.
    """
    models = model_path and [model_path] or MODELS_WITH_DENORM_FIELDS

    for model in models:
        check_model_denormalized_fields(model, fix=fix)
