from django.shortcuts import get_object_or_404

from rest_framework.views import APIView
from rest_framework.permissions import IsAuthenticated

from plan.common.utils.memoize import memoizable_property
from plan.common.utils.rest_fw import plan_response
from plan.services.api.team import DepartmentSerializer
from plan.services.models import Service
from plan.staff.constants import DEPARTMENT_ROLES
from plan.staff.models import Department, DepartmentStaff


class DepartmentView(APIView):
    permission_classes = (IsAuthenticated, )

    http_method_names = ('get',)
    _permissions_to_proceed = 'view_hierarchy'

    service_states = Service.states.ALL_STATES - {Service.states.CLOSED}

    def __init__(self, *args, **kwargs):
        self.serializer_class = DepartmentSerializer
        self.serializer_class.Meta.fields += ['level']

        super(DepartmentView, self).__init__(*args, **kwargs)

    def serialize_department(self, obj):
        serializer = self.serializer_class(obj)
        return serializer.data

    def get_access_context(self):
        sender = self.request.user.staff

        is_chief = (
            DepartmentStaff.objects
            .filter(
                department__in=self.department.get_ancestors(include_self=True),
                role=DEPARTMENT_ROLES.CHIEF,
                staff=sender,
            ).exists()
        )

        return {
            'show-access-tab': is_chief
        }

    def get(self, request, *args, **kwargs):
        department_dict = self.serialize_department(self.department)
        departments = [department_dict]
        #
        # Добавляем дочерние подразделения
        for child in self.department.children.active():
            departments.append(self.serialize_department(child))

        breadcrumbs = [self.serialize_department(department) for department in self.department.get_ancestors()]

        context = {
            # Должен быть один department, а дочерние внутри него
            'departments': departments,
            # Это должно быть тоже внутри department
            'breadcrumbs': breadcrumbs,
        }

        context.update(self.get_access_context())
        return plan_response(context)

    @memoizable_property
    def department(self):
        department_id = self.kwargs.get('id')

        if department_id:
            department = get_object_or_404(
                Department.objects.active(),
                pk=department_id
            )
        else:
            department_slug = self.kwargs.get('department_slug')

            if department_slug:
                department = get_object_or_404(
                    Department.objects.active(),
                    url=department_slug
                )
            else:
                # SenderMixin
                department = self.request.user.staff.department

        return department
