from django import forms
from django.shortcuts import get_object_or_404

from rest_framework.views import APIView
from rest_framework.permissions import IsAuthenticated

from plan.common.utils.intranet import departments
from plan.common.utils.rest_fw import plan_response
from plan.services.api.team import (
    PersonWithABCExtSerializer,
    PersonWithRoleSerializer,
    DepartmentSerializer
)
from plan.services.models import Service
from plan.services.helpers import guess_default_service_role
from plan.staff.models import Department, Staff


class DepartmentMembersForm(forms.Form):
    service = forms.ModelChoiceField(Service.objects.all(), required=False)
    default_role_for = forms.ChoiceField(required=False, choices=[
        ('service', '')
    ])


class DepartmentMembersView(APIView):
    permission_classes = (IsAuthenticated, )

    def get(self, request, id):
        department = get_object_or_404(Department.objects.active(), pk=id)
        persons = Staff.objects.filter(
            departments.get_members_query(
                department=department,
                with_nested=True,
            ),
            is_dismissed=False,
        )
        form = DepartmentMembersForm(request.GET)
        person_serializer_class = PersonWithABCExtSerializer

        prefetched_data = {}
        service_id = None
        if form.is_valid():
            if form.cleaned_data['service']:
                service_id = form.cleaned_data['service'].id

            if form.cleaned_data['default_role_for']:
                entity = form.cleaned_data['default_role_for']
                # TODO: сейчас рассматривается только entity == service, нужно продумать serializer для других
                PersonWithRoleSerializer.get_role_spec(entity)
                person_serializer_class = PersonWithRoleSerializer

                if entity == 'service':
                    default_roles = guess_default_service_role(
                        person_ids=[p.id for p in persons],
                        service_id=service_id,
                        individual_results=True,
                    )
                    prefetched_data['default_service_roles'] = default_roles

        return plan_response({
            'members': [person_serializer_class(person, prefetched_data=prefetched_data).data for person in persons],
            'departments': DepartmentSerializer(department).data,
        })
