import django_filters

from plan.api.filters import (
    BooleanFilter,
    CharInFilter,
    IntegerListFilter,
    CustomModelMultipleChoiceFilter,
    PlanFilterSet,
)
from plan.duty.models import Gap, Shift, Schedule
from plan.services.models import Service, ServiceMember
from plan.staff.models import Staff


class ScheduleFilterSet(PlanFilterSet):
    show_in_staff = BooleanFilter()

    class Meta:
        model = Schedule
        fields = {
            'id': ['exact'],
            'service': ['exact', 'in'],
            'service__slug': ['exact', 'in'],
            'show_in_staff': ['exact'],
        }


class GapFilterSet(PlanFilterSet):
    person = CustomModelMultipleChoiceFilter(
        field_name='staff__login',
        to_field_name='login',
        queryset=Staff.objects.all(),
    )
    service = CustomModelMultipleChoiceFilter(
        field_name='service',
        queryset=Service.objects.all(),
        lookup_expr='in',
        method='filter_services',
        distinct=False,
    )
    date_from = django_filters.DateTimeFilter(field_name='end', lookup_expr='gte', required=True)
    date_to = django_filters.DateTimeFilter(field_name='start', lookup_expr='lte', required=True)

    def filter_services(self, queryset, name, value):
        if not value:
            return queryset

        staff_pks = ServiceMember.objects.filter(service__in=value).values_list('staff', flat=True)
        return queryset.filter(staff__in=staff_pks)

    class Meta:
        model = Gap
        fields = ('date_from', 'date_to', 'person', 'service',)


class ShiftBaseFilterSet(PlanFilterSet):
    schedule__in = IntegerListFilter(field_name='schedule', lookup_expr='in',)

    schedule__slug = CustomModelMultipleChoiceFilter(
        field_name='schedule__slug',
        queryset=Schedule.objects.all(),
        to_field_name='slug'
    )

    schedule__slug__in = CharInFilter(
        field_name='schedule__slug',
        lookup_expr='in',
    )

    person = CustomModelMultipleChoiceFilter(
        field_name='staff__login',
        to_field_name='login',
        queryset=Staff.objects.all()
    )
    person__in = CharInFilter(field_name='staff__login', lookup_expr='in')

    has_problems = BooleanFilter()

    service = CustomModelMultipleChoiceFilter(
        field_name='schedule__service',
        queryset=Service.objects.all()
    )

    service__slug = CustomModelMultipleChoiceFilter(
        field_name='schedule__service__slug',
        queryset=Service.objects.all(),
        to_field_name='slug'
    )

    class Meta:
        model = Shift
        fields = ('has_problems', 'person', 'person__in',
                  'schedule', 'schedule__in', 'schedule__slug', 'schedule__slug__in',
                  'service', 'service__slug',)


class ShiftFilterSet(ShiftBaseFilterSet):
    date_from = django_filters.DateTimeFilter(field_name='end', lookup_expr='gte', required=True)
    date_to = django_filters.DateTimeFilter(field_name='start', lookup_expr='lte', required=True)

    class Meta:
        model = Shift
        fields = ShiftBaseFilterSet.Meta.fields + ('date_from', 'date_to',)
