from django.conf import settings
from rest_framework.permissions import BasePermission

from plan.api.permissions import TvmAuthenticated
from plan.duty.api.serializers import (
    DutyPermissionServiceSerializer,
    DutyPermissionShiftSerializer,
    DutyOnlyPersonShiftSerializer,
)
from plan.duty.permissions import is_watcher_tvm_user


def can_modify_shift(shift, person):
    return person.can_modify_duty_in_service(shift.service)


class DutyPermission(TvmAuthenticated):
    @staticmethod
    def _get_request_service(request):
        serializer = DutyPermissionServiceSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        return serializer.validated_data['service']

    def has_permission(self, request, view):
        if not super(DutyPermission, self).has_permission(request, view):
            return False

        ticket = getattr(request, 'tvm_service_ticket', None)
        if ticket is not None and is_watcher_tvm_user(ticket):
            return True

        if view.action in ('create',):
            service = self._get_request_service(request)
            return service is not None and request.person.can_modify_duty_in_service(service)

        return True

    def has_object_permission(self, request, view, obj):
        if not super(DutyPermission, self).has_permission(request, view):
            return False

        ticket = getattr(request, 'tvm_service_ticket', None)
        if ticket is not None and is_watcher_tvm_user(ticket):
            return True

        if view.action in ('update', 'partial_update', 'destroy'):
            return request.person.can_modify_duty_in_service(obj.service)

        return True


class DutyShiftPermission(DutyPermission):
    @staticmethod
    def _get_request_service(request):
        serializer = DutyPermissionShiftSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        replace_for = serializer.validated_data['replace_for']

        return replace_for.service

    @staticmethod
    def _can_approve_own_shifts(request, obj):
        if set(request.data.keys()) != {'is_approved'}:
            return False

        return request.person.staff == obj.staff

    @staticmethod
    def _can_change_person_on_self(request, obj):
        if set(request.data.keys()) not in (
            {'person', 'is_approved'},
            {'person'},
        ):
            return False
        serializer = DutyOnlyPersonShiftSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        return (
            serializer.validated_data['person'] == request.person.staff and
            request.person.is_schedule_member(obj.schedule)
        )

    def has_object_permission(self, request, view, obj):
        return (
            super(DutyShiftPermission, self).has_object_permission(request, view, obj) or
            self._can_change_person_on_self(request, obj) or
            self._can_approve_own_shifts(request, obj)
        )


class DutyToWatcherPermission(BasePermission):
    def has_permission(self, request, view):
        ticket = getattr(request, 'tvm_service_ticket', None)
        return ticket is not None and str(ticket.src) == settings.WATCHER_TVM_ID
