from collections import defaultdict, Counter

from django.core.validators import MinValueValidator
from rest_framework import serializers

from django.conf import settings
from django.db.models.query import QuerySet
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from plan import exceptions
from plan.api.exceptions import BadRequest, ValidationError
from plan.api.fields import IntegerInterfacedSerializerField, SlugInterfacedSerializerField
from plan.api.serializers import (
    CompactRoleSerializer,
    CompactServiceSerializer,
    CompactStaffSerializer,
    ModelSerializer,
)
from plan.duty.tracker import (
    check_can_grant_queue,
    enable_auto_assign,
    remove_component_lead,
    set_component_lead,
)
from plan.common.utils import timezone as utils
from plan.common.utils.timezone import make_localized_datetime
from plan.duty.api.validators import (
    validate_duty_duration,
    validate_role_of_service,
    validate_unique_slug,
    validate_unique_name,
    validate_forbid_update_slug,
    RoleValidator,
    StartDatetimeEndDatetimeValidator,
    StartWithValidator,
)
from plan.duty.models import Gap, Schedule, Shift, Order, DutyToWatcher
from plan.duty.permissions import is_watcher_tvm_user
from plan.duty.tasks import priority_recalculate_shifts_for_service, notify_staff_duty
from plan.idm.exceptions import IDMError
from plan.roles.models import Role
from plan.services.models import Service, ServiceMember, ServiceTag
from plan.staff.models import Staff


class DutyStaffSerializer(CompactStaffSerializer):
    vteams = serializers.SerializerMethodField()

    def __init__(self, *args, **kwargs):
        super(DutyStaffSerializer, self).__init__(*args, **kwargs)

    def get_vteams(self, instance):
        _vteams_cache = None
        view = self.context.get('view')
        cache_obj = view or self.parent
        if cache_obj:
            _vteams_cache = getattr(cache_obj, '_vteams_cache', None)
            if _vteams_cache is None and isinstance(self.parent.instance, QuerySet):
                _vteams_cache = defaultdict(set)
                staffs = [shift.staff for shift in self.parent.instance]
                members = (
                    ServiceMember.objects
                    .filter(staff__in=staffs, service__tags__name=ServiceTag.VTEAM_NAME)
                    .select_related('service')
                )
                for member in members:
                    _vteams_cache[member.staff_id].add(member.service)
                for staff in staffs:
                    if staff is not None and staff.pk not in _vteams_cache.keys():
                        _vteams_cache[staff.pk] = set()

                for k in _vteams_cache.keys():
                    _vteams_cache[k] = CompactServiceSerializer(_vteams_cache[k], many=True).data

        if _vteams_cache is None:
            _vteams_cache = defaultdict(set)

        if instance.pk not in _vteams_cache.keys():
            vteams = Service.objects.filter(members__staff=instance, tags__name=ServiceTag.VTEAM_NAME).distinct()
            _vteams_cache[instance.pk] = CompactServiceSerializer(vteams, many=True).data

        if cache_obj:
            setattr(cache_obj, '_vteams_cache', _vteams_cache)
        return _vteams_cache[instance.pk]

    class Meta:
        model = Staff
        fields = CompactStaffSerializer.Meta.fields + ['vteams']


class GapSerializer(ModelSerializer):
    person = CompactStaffSerializer(source='staff')

    class Meta:
        model = Gap
        fields = ['id', 'person', 'type', 'start', 'end', 'full_day', 'work_in_absence']


class OrderSerializer(ModelSerializer):
    person = DutyStaffSerializer(source='staff')
    order = serializers.IntegerField(read_only=True)

    class Meta:
        model = Order
        fields = ('person', 'order',)


class CalendarScheduleSerializer(ModelSerializer):
    id = serializers.IntegerField(required=False, min_value=1)
    role = IntegerInterfacedSerializerField(
        queryset=Role.objects.all(),
        serializer=CompactRoleSerializer,
        allow_null=True,
    )
    algorithm = serializers.ChoiceField(
        choices=Schedule.ALGORITHM,
        default=Schedule.NO_ORDER,
    )

    orders = SlugInterfacedSerializerField(
        many=True, required=False,
        queryset=Staff.objects.all(),
        slug_field='login',
        serializer=OrderSerializer,
    )

    start_with = SlugInterfacedSerializerField(
        queryset=Staff.objects.all(),
        slug_field='login',
        serializer=DutyStaffSerializer,
        required=False,
        write_only=True,
    )
    slug = serializers.SlugField(read_only=True)

    class Meta:
        model = Schedule
        fields = (
            'id', 'name', 'role', 'persons_count', 'consider_other_schedules',
            'algorithm', 'orders', 'start_with', 'slug',
        )

    def validate(self, attrs):
        algorithm = attrs.get('algorithm', False)
        orders = attrs.get('orders', False)

        if (algorithm or orders) is not False:
            # валидируем только если хотя бы один из параметров был передан
            if self.instance:
                if algorithm is False:
                    algorithm = self.instance.algorithm
                if orders is False:
                    orders = [
                        obj.staff for obj in
                        self.instance.orders.select_related('staff').all()
                    ]

            if algorithm == Schedule.MANUAL_ORDER:
                if not orders:
                    raise ValidationError(
                        {'orders': f'orders list cannot be empty if algorithm is set to {Schedule.MANUAL_ORDER}'}
                    )
                else:
                    duplicated_logins = [login for login, count in Counter(o.login for o in orders).items() if
                                         count > 1]
                    if duplicated_logins:
                        raise ValidationError({'orders': f'logins {",".join(duplicated_logins)} are duplicated'})

        return attrs


def is_data_no_order_consider(data):
    if 'algorithm' in data:
        return data['algorithm'] == Schedule.NO_ORDER and data.get('consider_other_schedules', True)
    if 'consider_other_schedules' in data:
        return data['consider_other_schedules']

    return False


class ScheduleSerializer(CalendarScheduleSerializer):

    duration = serializers.DurationField(validators=[validate_duty_duration])
    service = IntegerInterfacedSerializerField(
        queryset=Service.objects.all(),
        serializer=CompactServiceSerializer,
        allow_null=False,
    )
    slug = serializers.SlugField(
        allow_blank=False,
        max_length=settings.MAX_SLUG_LENGTH,
    )
    is_important = serializers.BooleanField(read_only=True, required=False)
    start_time = serializers.TimeField(required=False, input_formats=['%H:%M'], help_text='Формат %H:%M')
    show_in_staff = serializers.BooleanField(required=False)
    autoapprove_timedelta = serializers.DurationField(
        required=False,
        validators=[MinValueValidator(timezone.timedelta(days=0))]
    )

    # start_with наследуем от CalendarScheduleSerializer

    class Meta:
        model = Schedule
        fields = (
            'id', 'recalculate', 'name', 'description', 'role', 'persons_count', 'orders',
            'consider_other_schedules', 'algorithm', 'autoapprove_timedelta',
            'service', 'duration', 'only_workdays', 'start_date', 'start_time',
            'role_on_duty', 'slug', 'allow_sequential_shifts', 'show_in_staff',
            'days_for_problem_notification', 'days_for_begin_shift_notification',
            'duty_on_holidays', 'duty_on_weekends', 'start_with', 'tracker_queue_id',
            'tracker_component_id', 'is_important',
        )
        validators = [
            RoleValidator(),
        ]

    def validate_tracker_component(self, attrs):
        current_tracker_queue = None
        current_tracker_component = None

        persons_count = attrs.get('persons_count', getattr(self.instance, 'persons_count', 1))

        new_tracker_queue = attrs.get('tracker_queue_id', False)
        new_tracker_component = attrs.get('tracker_component_id', False)

        if self.instance is not None:
            current_tracker_queue = self.instance.tracker_queue_id
            current_tracker_component = self.instance.tracker_component_id

        if new_tracker_queue and not new_tracker_component:
            raise ValidationError(
                message={
                    'ru': 'Необходимо указать tracker_component_id если передан tracker_queue_id',
                    'en': 'You must specify tracker_component_id if tracker_queue_id is passed',
                }
            )

        if (
            new_tracker_component or
            (current_tracker_component and new_tracker_component is False)
        ) and persons_count > 1:
            raise ValidationError(
                message={
                    'ru': 'Нельзя указать, что дежурят несколько человек если tracker_component_id задан',
                    'en': 'You cannot set persons_count > 1 if tracker_component_id is set',
                }
            )

        if new_tracker_component != current_tracker_component:
            if new_tracker_component:
                # значение выставляют/меняют на другое
                already_used = Schedule.objects.active().filter(tracker_component_id=new_tracker_component).first()
                if already_used:
                    raise ValidationError(
                        message={
                            'ru': f'Этот компонент уже используется в другом расписании '
                                  f'service_id: {already_used.service_id}, schedule_id: {already_used.id}',
                            'en': f'This component is already used in different schedule - '
                                  f'service_id: {already_used.service_id}, schedule_id: {already_used.id}',
                        }
                    )

                can_grant_queue = check_can_grant_queue(queue_id=new_tracker_queue, user_ticket=self.context['request'].tvm_user_ticket)
                if not can_grant_queue:
                    raise ValidationError(
                        message={
                            'ru': 'У пользователя отсутствуют grant права на данную очередь',
                            'en': 'Grant permissions for chosen queue is missing for current user',
                        }
                    )
                enable_auto_assign(component_id=new_tracker_component, user_ticket=self.context['request'].tvm_user_ticket)
                if self.instance:
                    current_duty = self.instance.shifts.with_staff().current_shifts().started().select_related('staff').first()
                    if current_duty:
                        set_component_lead(component_id=new_tracker_component, staff=current_duty.staff)
            elif current_tracker_component and new_tracker_component is not False:
                # значение очищают
                can_grant_queue = check_can_grant_queue(queue_id=current_tracker_queue, user_ticket=self.context['request'].tvm_user_ticket)
                if not can_grant_queue:
                    raise ValidationError(
                        message={
                            'ru': 'У пользователя должны быть grant права на очередь, для того чтобы очистить это поле',
                            'en': 'You must have grant permission on queue in order to erase this field',
                        }
                    )
                remove_component_lead(component_id=current_tracker_component)

    def validate(self, attrs):
        service = attrs.get('service')
        if service is None and self.instance is not None:
            service = self.instance.service

        validate_role_of_service(attrs, service)
        tvm_service_ticket = getattr(self.context.get('request'), 'tvm_service_ticket', None)
        if self.instance is not None and not is_watcher_tvm_user(ticket=tvm_service_ticket):
            validate_forbid_update_slug(attrs, self.instance)
        attrs = super(ScheduleSerializer, self).validate(attrs)
        StartWithValidator(attrs, self.instance)()
        self.validate_tracker_component(attrs)
        return attrs

    def create(self, validated_data):
        service = validated_data.pop('service')
        self.validate_unique_slug_and_name(validated_data, service)
        schedule = Schedule.objects.add_or_update_schedule(service, validated_data)
        full_recalculation_ids = []
        if schedule.is_no_order_consider():
            full_recalculation_ids = schedule.get_associated_schedules_id_list(include_self=True)
        priority_recalculate_shifts_for_service.apply_async_on_commit(args=[service.id, full_recalculation_ids])
        if service.flags and not service.flags.get('duty1'):
            service.flags = {
                'duty1': True,
                'duty2': False,
            }
            service.save()
        return schedule

    def update(self, instance, validated_data):
        service = validated_data.pop('service', instance.service)

        self.validate_unique_slug_and_name(validated_data, service, instance)

        need_update_shifts_datetime = (
            'start_time' in validated_data
            and validated_data['start_time'] != instance.start_time
        )
        noorder_state_changed = False
        if any(field in validated_data for field in ('consider_other_schedules', 'algorithm')):
            noorder_state_changed = instance.is_no_order_consider() != is_data_no_order_consider(validated_data)

        # проадейтим инстанс и посчитаем критические поля
        instance, critical_fields = self.update_instance_and_critical_fields(instance, validated_data, service)
        # даже если удалили все смены в будущем, текущие нужно проапдейтить
        if need_update_shifts_datetime:
            instance.update_shifts_date()

        full_recalculation_ids = []
        if noorder_state_changed or (instance.is_no_order_consider() and critical_fields):
            full_recalculation_ids = instance.get_associated_schedules_id_list()
        priority_recalculate_shifts_for_service.apply_async_on_commit(
            args=[instance.service.id, full_recalculation_ids]
        )
        return instance

    def validate_unique_slug_and_name(self, validated_data, service, instance=None):
        schedules_to_delete = []
        schedules_to_delete += validate_unique_slug(validated_data, service, instance)
        schedules_to_delete += validate_unique_name(validated_data, service, instance)
        for schedule in schedules_to_delete:
            schedule.delete()

    def update_instance_and_critical_fields(self, instance, validated_data, service):
        """
        Делает апдейт instance и проверяет переданные поля на критичность.
        Если изменились критичные поля, то нужно запустить полный пересчёт с удалением будующих смен.
        При этом важно не запустить его, если поля переданы, но изменений не было.
        """
        # нужно запомнить offset и orders до апдейта
        old_offset = instance.manual_ordering_offset
        old_orders = None
        if validated_data.get('orders') or validated_data.get('start_with'):
            old_orders = dict()
            for order in instance.orders.order_by('order'):
                login = order.staff.login if order.staff else None
                old_orders[login] = order.order

        critical_fields = self.get_changed_crit_fields(instance, validated_data)
        instance = Schedule.objects.add_or_update_schedule(service, validated_data.copy(), instance.pk)
        new_offset = instance.manual_ordering_offset
        if instance.algorithm == Schedule.MANUAL_ORDER:
            critical_fields = self.get_changed_crit_fields_manual_order(
                instance, validated_data, old_orders, old_offset, new_offset, critical_fields
            )
        if critical_fields:
            instance.shifts.future().delete()

        return instance, critical_fields

    def get_changed_crit_fields(self, instance, validated_data):
        critical_fields = [
            'duration',
            'only_workdays',
            'duty_on_holidays',
            'duty_on_weekends',
            'start_date',
            'algorithm',
            'role',
        ]
        if 'persons_count' in validated_data and (
            validated_data['persons_count'] < instance.persons_count or
            validated_data['persons_count'] > instance.persons_count and instance.algorithm == Schedule.MANUAL_ORDER
        ):
            critical_fields.append('persons_count')

        if instance.allow_sequential_shifts:
            critical_fields.append('allow_sequential_shifts')

        return [
            field
            for field in critical_fields
            if field in validated_data and getattr(instance, field) != validated_data[field]
        ]

    def get_changed_crit_fields_manual_order(self, instance, validated_data,
                                             current_orders, old_offset, offset,
                                             critical_fields):
        """
        Изменение порядка для графиков с ручным порядком проверяем отдельно от всех остальных критических полей
        """
        orders = validated_data.get('orders')
        start_with = validated_data.get('start_with')

        if old_offset is None or (
                current_orders is None and
                'algorithm' in validated_data
                and instance.algorithm != validated_data['algorithm']
        ):
            # то порядка раньше не было (иначе offset был бы 0 как минимум и существовал бы current_orders),
            # если алгоритм уже изменили, значит валидацию уже прошли и orders там есть точно
            critical_fields.append('orders')

        elif orders:
            orders_as_they_are = []
            if len(current_orders.keys()) != len(orders):
                critical_fields.append('orders')

            elif start_with is None:
                for staff in validated_data['orders']:
                    if staff.login in current_orders and current_orders[staff.login] == len(orders_as_they_are):
                        orders_as_they_are.append(current_orders[staff.login])
                    else:
                        # порядок уже не сходится, тк либо отсутствует логин в текущем порядке,
                        # либо не в том порядке, поэтому можно не продолжать
                        critical_fields.append('orders')
                        break

            else:
                len_orders = len(orders)  # выше уже выяснили, что длины совпадают
                for order, staff in enumerate(validated_data['orders']):
                    if staff.login not in current_orders:
                        break

                    if current_orders[staff.login] != (order - offset) % len_orders:
                        critical_fields.append('orders')
                        break

        elif start_with:
            # если нет orders => используем старый порядок,
            # если оффсеты равны => изменений не было
            # также равенство оффсетов будет, если новый == старый - длина
            offset = instance.manual_ordering_offset
            if not (offset == old_offset or offset == old_offset - len(current_orders.keys())):
                critical_fields.append('orders')

        return critical_fields


class ShiftBaseSerializer(ModelSerializer):
    person = SlugInterfacedSerializerField(
        queryset=Staff.objects.all(),
        slug_field='login',
        source='staff',
        serializer=DutyStaffSerializer,
    )
    problems_count = serializers.SerializerMethodField()
    schedule = IntegerInterfacedSerializerField(
        queryset=Schedule.objects.all(),
        serializer=CalendarScheduleSerializer,
        required=False,
    )

    # ToDo: убрать, когда будем удалять поля start/end
    start_datetime = serializers.DateTimeField(required=False)
    end_datetime = serializers.DateTimeField(required=False)

    class Meta:
        model = Shift
        fields = (
            'id', 'person', 'problems_count', 'schedule', 'is_approved',
            'start', 'end', 'start_datetime', 'end_datetime',
        )

    def to_representation(self, instance):
        ret = super(ShiftBaseSerializer, self).to_representation(instance)
        ret['start_datetime'] = timezone.localtime(instance.start_datetime).isoformat()
        ret['end_datetime'] = timezone.localtime(instance.end_datetime).isoformat()
        return ret

    def get_problems_count(self, obj):
        return 1 if obj.has_problems else 0

    def validate(self, attrs):
        # Через api можно создавать только замены или загружать историю
        attrs = super(ShiftBaseSerializer, self).validate(attrs)
        if not self.have_schedule(attrs):
            return attrs
        self.fill_date_fields(attrs)
        return attrs

    def fill_date_fields(self, attrs):
        # Должны быть заполнены или поля (start, end) или (start_datetime, end_datetime), но не все сразу
        if ('end' in attrs or 'start' in attrs) and ('start_datetime' in attrs or 'end_datetime' in attrs):
            raise ValidationError(
                _('Должны быть заполнены одни из полей (start, end) или (start_datetime, end_datetime)')
            )
        if 'end' in attrs:
            # Из-за добавления времени начала/окончания дежурства, текущая смена фактически завершается не в end_date,
            # а на следующий день в start_time, поэтому нам нужно прибавить к дате 1 день
            attrs['end_datetime'] = (
                make_localized_datetime(attrs['end'], self.start_time(attrs))
                + timezone.timedelta(days=1)
            )
        if 'start' in attrs:
            attrs['start_datetime'] = make_localized_datetime(attrs['start'], self.start_time(attrs))
        if 'start_datetime' in attrs:
            attrs['start'] = attrs['start_datetime'].date()
        if 'end_datetime' in attrs:
            # end_date может отличаться от end_datetime,
            # например, если start_time - 00:00
            attrs['end'] = (attrs['end_datetime'] - timezone.timedelta(minutes=1)).date()
        return attrs

    def have_schedule(self, attrs):
        result = False
        if self.instance:
            result = True
        elif 'replace_for' in attrs:
            result = True
        elif 'schedule' in attrs:
            result = True
        return result

    def start_time(self, attrs):
        if not hasattr(self, '_start_time'):
            if self.instance:
                self._start_time = self.instance.schedule.start_time
            elif 'replace_for' in attrs:
                self._start_time = attrs['replace_for'].schedule.start_time
            else:
                self._start_time = attrs['schedule'].start_time
        return self._start_time


class ShiftUploadHistorySerializer(ShiftBaseSerializer):
    start_datetime = serializers.DateTimeField(
        required=True,
        help_text="Например, '2021-01-01T13:00:00'",
    )
    end_datetime = serializers.DateTimeField(
        required=True,
        help_text="Например, '2021-01-18T19:00:00'"
    )

    class Meta:
        model = Shift
        fields = (
            'id', 'schedule',
            'person', 'is_approved',
            'start_datetime', 'end_datetime',
        )
        validators = [
            StartDatetimeEndDatetimeValidator(),
        ]


class ShiftOptionalUpdateSerializer(ShiftBaseSerializer):
    id = serializers.IntegerField(required=False)


class ShiftSerializer(ShiftBaseSerializer):
    replace_for = ShiftBaseSerializer(read_only=True,)
    replaces = ShiftOptionalUpdateSerializer(many=True, read_only=False, required=False)
    schedule = CalendarScheduleSerializer(read_only=True,)

    class Meta(ShiftBaseSerializer.Meta):
        fields = ShiftBaseSerializer.Meta.fields + ('replaces', 'replace_for')
        read_only_fields = ('start', 'end', 'start_datetime', 'end_datetime')

    def update(self, instance, validated_data):
        from plan.duty.schedulers.scheduler import DutyScheduler

        staff = validated_data.pop('staff', None)
        new_replaces = validated_data.pop('replaces', None)
        is_approved = validated_data.pop('is_approved', None)
        instance = super(ShiftSerializer, self).update(instance, validated_data)
        need_recalculate = False

        if staff is not None and staff != instance.staff:
            need_recalculate = True
            try:
                DutyScheduler.update_shift_staff(instance, staff, save=True, send_email=True)

            except IDMError:
                raise BadRequest('Wrong IDM answer')

        if new_replaces is not None:
            need_recalculate = True
            self.fill_date_in_replaces(new_replaces)
            self.fill_role_in_replaces(new_replaces, instance.schedule.get_role_on_duty())
            try:
                instance.update_replaces(new_replaces, self.context['request'].person.staff)

            except exceptions.DataValidationError:
                raise serializers.ValidationError('Задан некорректный интервал для замены.')

        # Меняем is_approved в двух случаях:
        #   * это явно указали в patch'е
        #   * сделали замену (полную или частичную)
        if (
            is_approved is not None and is_approved != instance.is_approved
            or is_approved is None and need_recalculate
        ):
            if is_approved is None:
                is_approved = True
            instance.set_approved(is_approved, self.context['request'].person.staff)
            notify_staff_duty.delay(instance.service.id, instance.service.slug)

        if need_recalculate:
            scheduler = DutyScheduler.initialize_scheduler_by_shift(instance)
            # пересчет календаря происходит только после полной или временной замены
            if instance.schedule.algorithm == Schedule.MANUAL_ORDER:
                scheduler.update_shift_problems(instance, save_if_updated=True)
            else:
                # сначала проверим текущий шифт, чтобы приблизить к правде то, что придет пользователю,
                # а уже потом запустим пересчет всего календаря
                scheduler.check_one_shift(instance, instance.schedule)
                priority_recalculate_shifts_for_service.apply_async(args=[instance.service.id])

        return instance

    def fill_date_in_replaces(self, replaces):
        for replace in replaces:
            self.fill_date_fields(replace)

    def fill_role_in_replaces(self, replaces, role):
        for replace in replaces:
            replace['role'] = role


class ShiftReadSerializer(ShiftSerializer):
    person = DutyStaffSerializer(source='staff')


class ShiftCreateSerializer(ShiftSerializer):
    replace_for = IntegerInterfacedSerializerField(
        queryset=Shift.objects.filter(replace_for=None),
        serializer=ShiftBaseSerializer,
        required=True,
    )

    # ToDo: убрать, когда будем удалять поля start/end
    start_datetime = serializers.DateTimeField(
        required=True,
        help_text="Например, '2021-01-01T13:00:00'",
    )
    end_datetime = serializers.DateTimeField(
        required=True,
        help_text="Например, '2021-01-18T19:00:00'"
    )

    class Meta:
        model = Shift
        fields = ('id', 'person', 'start_datetime', 'end_datetime', 'is_approved', 'schedule', 'replaces', 'replace_for',)

    def create(self, validated_data):
        start_new = validated_data['start_datetime']
        end_new = validated_data['end_datetime']
        start_old = validated_data['replace_for'].start_datetime
        end_old = validated_data['replace_for'].end_datetime
        today = utils.today()
        all_intersecting_replacements = Shift.objects.filter(
            replace_for=validated_data['replace_for'],
            start_datetime__lte=end_new,
            end_datetime__gte=start_new,
        )

        if not start_old <= start_new <= end_new <= end_old:
            raise serializers.ValidationError('Задан некорректный интервал для замены.')

        if all_intersecting_replacements.exists():
            replace = all_intersecting_replacements.first()
            raise serializers.ValidationError(
                'Даты пересекаются с частичной заменой {login} c {start} по {end}.'
                .format(login=replace.staff.login, start=replace.start, end=replace.end)
                )

        schedule = validated_data['replace_for'].schedule
        validated_data['schedule'] = schedule
        validated_data['role'] = schedule.get_role_on_duty()
        validated_data['is_approved'] = True
        validated_data['approved_by'] = self.context['request'].person.staff
        replacement = Shift.objects.create(**validated_data)
        if replacement.start <= today <= replacement.end:
            replacement.begin()
            replacement.save()

        priority_recalculate_shifts_for_service.apply_async(args=[replacement.service.id])
        return replacement


class DutyPermissionServiceSerializer(serializers.Serializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.alive(),
        required=True,
    )


class DutyPermissionShiftSerializer(serializers.Serializer):
    replace_for = serializers.PrimaryKeyRelatedField(
        queryset=Shift.objects.filter(replace_for=None),
        required=True,
    )


class DutyOnlyPersonShiftSerializer(serializers.Serializer):
    person = serializers.SlugRelatedField(
        queryset=Staff.objects.all(),
        slug_field='login',
    )


class AllowForDutyRequestSerializer(serializers.Serializer):
    schedule = serializers.PrimaryKeyRelatedField(
        queryset=Schedule.objects.all(),
        required=False,
    )
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.all(),
        required=True,
    )
    role = serializers.PrimaryKeyRelatedField(
        queryset=Role.objects.all(),
        many=True,
        required=False,
    )
    ordered = serializers.BooleanField(required=False)

    def validate(self, data):
        if data.get('ordered') and not data.get('schedule'):
            raise ValidationError({
                'ordered': 'ordered set without schedule'
            })
        return data


class AllowForDutyResponseSerializer(CompactStaffSerializer):

    order = serializers.SerializerMethodField(read_only=True)
    active_duty = serializers.SerializerMethodField(read_only=True)
    start_with = serializers.SerializerMethodField(read_only=True)

    class Meta:
        model = CompactStaffSerializer.Meta.model
        fields = CompactStaffSerializer.Meta.fields + ['order', 'active_duty', 'start_with']

    def __init__(self, *args, **kwargs):
        schedule = kwargs['context'].pop('schedule', None)
        super(AllowForDutyResponseSerializer, self).__init__(*args, **kwargs)
        self._schedule = schedule
        self._start_with = None
        if schedule:
            self._start_with = schedule.get_start_with()

    def get_order(self, staff):
        return self._schedule and self._schedule.get_staff_order(staff)

    def get_active_duty(self, staff):
        return self._schedule is not None and self._schedule.is_staff_active_duty(staff)

    def get_start_with(self, staff):
        return self._start_with == staff


class FrontendScheduleSerializer(ScheduleSerializer):
    requester_in_duty = serializers.BooleanField(read_only=True)

    class Meta(ScheduleSerializer.Meta):
        fields = ScheduleSerializer.Meta.fields + ('requester_in_duty',)


class DutyToWatcherSerializer(ModelSerializer):
    class Meta:
        model = DutyToWatcher
        fields = ['abc_id', 'watcher_id']
