import logging

from django.utils.translation import ugettext_lazy as _
from rest_framework.exceptions import ValidationError
from rest_framework.generics import get_object_or_404

from plan.common.utils.watcher import WatcherClient
from plan.api.exceptions import BadRequest
from plan.duty.constants import WARNING_CONDITIONS, WARNING_CONSEQUENCES
from plan.duty.models import Schedule
from plan.duty.schedulers import ManualOrderingScheduler
from plan.roles.models import Role


log = logging.getLogger(__name__)


# Сейчас длительность дежурства должна быть кратна дням
def validate_duty_duration(value):
    if value.total_seconds() % (24 * 3600) != 0:
        raise ValidationError('This field must be "DD 00"')


def validate_role_of_service(schedule_data, service):
    role_on_duty = schedule_data.get('role_on_duty', None)

    if role_on_duty is not None and role_on_duty.service is not None and role_on_duty.service != service:
        raise ValidationError({
            'role_on_duty': _('Роль %s не может быть выдана во время дежурства. Роль не относится к сервису') % role_on_duty
        })


def validate_unique_slug(schedule_data, service, instance=None):
    slug = schedule_data.get('slug', None)

    if instance is not None and slug == instance.slug:
        return []

    if slug is not None:
        watcher_client = WatcherClient()
        watcher_schedules = watcher_client.get_schedules(service_id=service.id)
        for obj in watcher_schedules:
            if obj['slug'] == slug:
                raise ValidationError({'slug': _('Слаг %s не уникален') % slug})

        existing_schedules = service.schedules.filter(slug=slug)

        if existing_schedules.exists():
            schedule = existing_schedules.first()

            if schedule.status == Schedule.DELETED:
                return [schedule]
            else:
                raise ValidationError({'slug': _('Слаг %s не уникален') % slug})

    return []


def validate_unique_name(schedule_data, service, instance=None):
    name = schedule_data.get('name', None)

    if instance is not None and name == instance.name:
        return []

    if name is not None:
        existing_schedules = service.schedules.filter(name=name)

        if existing_schedules.exists():
            schedule = existing_schedules.first()

            if schedule.status == Schedule.DELETED:
                return [schedule]
            else:
                raise ValidationError({'name': _('Название %s не уникально') % name})

    return []


def validate_forbid_update_slug(schedule_data, instance):
    if 'slug' in schedule_data and instance.slug != '' and instance.slug != schedule_data['slug']:
        raise ValidationError({'slug': _('Слаг не может быть изменен')})


class RoleValidator(object):
    requires_context = True

    def __call__(self, attrs, instance, *args, **kwargs):
        role = attrs.get('role', getattr(instance, 'role', None))
        role_on_duty = attrs.get('role_on_duty', getattr(instance, 'role_on_duty', None))
        if role_on_duty is None:
            role_on_duty = Role.objects.globalwide().get(code=Role.DUTY)

        if (
            role == role_on_duty
            or role_on_duty.code in Role.CAN_NOT_USE_FOR_DUTY
            or not role_on_duty.scope.can_issue_at_duty_time
        ):
            raise ValidationError({
                'role_on_duty': _('Роль %s не может быть выдана во время дежурства') % role_on_duty.name
            })


class StartWithValidator:
    def __init__(self, schedule_data, schedule):
        self.schedule_data = schedule_data
        self.schedule = schedule

    def __call__(self):
        log.info(f'schedule_data: {self.schedule_data}')
        start_with = self.schedule_data.get('start_with', None)

        if start_with is None:
            return self.schedule_data

        staffs = self.schedule_data.get('orders')
        orders = set()
        if staffs:
            for staff in staffs:
                orders.add(staff.login)

        if self.schedule is not None and len(orders) == 0:
            orders = self.schedule.orders.values_list('staff__login', flat=True)

        log.info(f'Order in schedule: {orders}')
        if not orders or start_with.login not in orders:
            raise BadRequest('start_with not in orders')

        return self.schedule_data


class StartDatetimeEndDatetimeValidator:
    """
    Проверяем, что начало смены раньше окончания.
    """

    def __call__(self, attrs, *args, **kwargs):
        start_datetime = attrs.get('start_datetime')
        end_datetime = attrs.get('end_datetime')

        if end_datetime < start_datetime:
            raise BadRequest(f'Shift date incorrect: end {end_datetime} before start {start_datetime}.')


class ScheduleFieldsChecker:
    """
    Класс для проверки изменения в расписании, используется для формирования предупреждения пользователю о том,
    что какие-то графики дежурств будут пересчитаны после изменения полей.
    """
    def __init__(self, request, serializer_class, queryset, view):
        self.schedule_changed = False
        self.fields = set()
        self.consequence = set()
        self.queryset = queryset
        self.serializer_class = serializer_class
        self.request = request
        self.view = view
        self.instance = None
        self.deleted_ids = set(self.get_deleted_schedule_ids())

    def get_consequence(self):
        if not self.consequence:
            return None
        if WARNING_CONSEQUENCES.HARD in self.consequence:
            return WARNING_CONSEQUENCES.HARD
        return WARNING_CONSEQUENCES.SOFT

    def get_conditions(self):
        conditions = []
        if self.fields:
            conditions.append(WARNING_CONDITIONS.FIELDS)
        if self.schedule_changed:
            conditions.append(WARNING_CONDITIONS.SCHEDULE)
        return conditions

    def get_deleted_schedule_ids(self):
        ids = (
            schedule['id']
            for schedule in self.request.data
            if schedule.get('status', None) == Schedule.DELETED
            and schedule.get('id', None)
        )
        return ids

    def validate(self):
        for schedule in self.request.data:
            self._check_one_obj(schedule)

    def _get_object_by_id(self, obj_id):
        obj = get_object_or_404(self.queryset, id=obj_id)
        self.view.check_object_permissions(self.request, obj)
        return obj

    def _check_one_obj(self, schedule):
        self.instance = None
        if schedule.get('id'):
            self.instance = self._get_object_by_id(schedule['id'])
        serializer = self.serializer_class(instance=self.instance, data=schedule, partial=True)
        serializer.is_valid(raise_exception=True)
        self.check_fields(serializer)

    def get_response_data(self):
        return {
            'conditions': self.get_conditions(),
            'fields': list(self.fields),
            'consequence': self.get_consequence(),
        }

    def check_fields(self, serializer):
        if self.instance is None:
            return self.check_new_obj(serializer)
        else:
            return self.check_update(serializer)

    def check_new_obj(self, serializer):
        # Если добавляется новый график и у сервиса есть графики с автоматическим порядком, которые учитывают другие,
        # то будет запущен пересчёт
        service_id = serializer.validated_data['service']
        if Schedule.objects.filter(service_id=service_id).no_order_consider().exclude(id__in=self.deleted_ids).exists():
            self.schedule_changed = True
            self.consequence.add(WARNING_CONSEQUENCES.SOFT)

    def check_update(self, serializer):
        has_other_considers_schedules = (
            Schedule.objects
            .filter(service_id=self.instance.service_id)
            .no_order_consider()
            .exclude(id__in=(self.instance.id, *self.deleted_ids,))
            .exists()
        )
        if not has_other_considers_schedules:
            return

        current_orders = None
        old_offset = None
        new_offset = None
        if 'order' in serializer.validated_data or 'start_with' in serializer.validated_data:
            for order in self.instance.orders.order_by('order'):
                login = order.staff.login if order.staff else None
                current_orders[login] = order.order

            old_offset = self.instance.manual_ordering_offset
            order = serializer.validated_data.get('order')
            start_with = serializer.validated_data.get('start_with')
            if start_with is None and order is not None:
                start_with = order[0]
            new_offset = ManualOrderingScheduler.get_offset(self, start_with)

        critical_fields = serializer.get_changed_crit_fields(self.instance, serializer.validated_data)
        algorithm = serializer.validated_data.get('algorithm')
        if (
            algorithm == Schedule.MANUAL_ORDER or
            (self.instance.algorithm == Schedule.MANUAL_ORDER and algorithm != Schedule.NO_ORDER)
        ):
            critical_fields = serializer.get_changed_crit_fields_manual_order(
                self.instance, serializer.validated_data, current_orders, old_offset, new_offset, critical_fields
            )

        if critical_fields:
            self.fields.update(critical_fields)
            self.consequence.add(WARNING_CONSEQUENCES.HARD)
        persons_count = serializer.validated_data.get('persons_count')
        if persons_count and persons_count > self.instance.persons_count:
            self.fields.add('persons_count')
            self.consequence.add(WARNING_CONSEQUENCES.SOFT)
        if serializer.validated_data.get('consider_other_schedules') is not None:
            self.fields.add('consider_other_schedules')
            self.consequence.add(WARNING_CONSEQUENCES.SOFT)
        if serializer.validated_data.get('id') in self.deleted_ids:
            self.schedule_changed = True
            self.consequence.add(WARNING_CONSEQUENCES.SOFT)
