from collections import OrderedDict
from dateutil import parser
from datetime import timedelta

from django.db import transaction
from django.conf import settings
from django.db.models import BooleanField, Case, Exists, OuterRef, Q, When
from django.utils import timezone
from drf_yasg.utils import swagger_auto_schema

from rest_framework import mixins, viewsets
from rest_framework.decorators import action
from rest_framework.response import Response

from plan.api.mixins import (
    DefaultFieldsMixin, DontFilterOnGetObjectMixin, NoPaginationListMixin,
    SelectOnlyFieldsMixin, TvmAccessMixin, OrderingMixin
)
from plan.api.base import ABCCursorPagination, ABCPagination
from plan.api.exceptions import BadRequest, PermissionDenied
from plan.common.utils import timezone as utils
from plan.duty.api.filters import GapFilterSet, ShiftFilterSet, ScheduleFilterSet, ShiftBaseFilterSet
from plan.duty.api.metadata import ShiftMetadata
from plan.duty.api.permissions import (
    DutyPermission,
    DutyShiftPermission,
    DutyToWatcherPermission,
)
from plan.duty.api.serializers import (
    GapSerializer,
    ShiftBaseSerializer,
    ShiftReadSerializer,
    ShiftSerializer,
    ShiftCreateSerializer,
    ScheduleSerializer,
    AllowForDutyRequestSerializer,
    AllowForDutyResponseSerializer,
    FrontendScheduleSerializer,
    ShiftUploadHistorySerializer,
    DutyToWatcherSerializer,
)
from plan.common.utils.watcher import WatcherClient
from plan.duty.api.validators import ScheduleFieldsChecker
from plan.duty.models import Gap, Shift, Schedule, Staff, Order, DutyToWatcher
from plan.duty.tasks import priority_recalculate_shifts_for_service
from plan.history.mixins import HistoryMixin
from plan.holidays.utils import date_is_holiday_type
from plan.idm.exceptions import IDMError
from plan.roles.models import Role
from plan.services.models import ServiceMember, Service
from plan.swagger import SwaggerDuty, SwaggerFrontend


def _add_watcher_shifts_data(watcher_shifts, result):
    staff_data = {
        staff.login: staff
        for staff in Staff.objects.filter(login__in=[
            shift['staff']['login'] for shift in watcher_shifts
            if shift.get('staff')
        ]
        )
    }
    for shift in watcher_shifts:
        staff = shift.get('staff')
        if not staff:
            continue

        staff_login = staff['login']
        staff_object = staff_data[staff_login]
        name = {
            'ru': f'{staff_object.first_name} {staff_object.last_name}'.strip(),
            'en': f'{staff_object.first_name_en} {staff_object.last_name_en}'.strip(),
        }

        end = parser.parse(shift['end'])
        if (0, 0) == (end.minute, end.hour):
            end = end - timedelta(days=1)
        end = end.date().isoformat()

        start = parser.parse(shift['start']).date().isoformat()

        result.append(
            {
                'id': shift['id'],
                'from_watcher': True,
                'is_primary': shift['is_primary'],
                'person': {
                    'id': staff['staff_id'],
                    'login': staff_login,
                    'name': name,
                },
                'schedule': {
                    'id': shift['schedule']['id'],
                    'name': shift['schedule']['name'],
                    'slug': shift['schedule']['slug'],
                },
                'is_approved': shift['approved'],
                'start': start,
                'end': end,
                'replaces': [],
                'start_datetime': shift['start'],
                'end_datetime': shift['end'],
            }
        )


class ScheduleView(HistoryMixin, viewsets.ModelViewSet):
    queryset = Schedule.objects.active()
    filter_class = ScheduleFilterSet
    permission_classes = (DutyPermission,)
    serializer_class = ScheduleSerializer
    _permissions_to_proceed = 'view_duty'

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['post'], detail=True)
    @transaction.atomic
    def upload_history(self, request, pk=None):
        """
        Старая ручка, поэтому не показываем в документации.
        Вместо неё есть две других:
            - /schedules/{id}/shifts/append/
            - /schedules/{id}/shifts/replace/
        """
        schedule = self.get_object()
        serializer = ShiftBaseSerializer(data=request.data, many=True)
        serializer.is_valid(raise_exception=True)
        requester = self.request.person

        if not requester.is_responsible(schedule.service):
            raise PermissionDenied('You are not responsible for service %s' % schedule.service.slug)

        upload_schedules = {obj['schedule'] for obj in serializer.validated_data}

        if len(upload_schedules) > 1 or schedule not in upload_schedules:
            raise BadRequest('Wrong schedules %s for this calendar' % upload_schedules)

        Shift.objects.filter(schedule=schedule).delete()
        serializer.save()

        if schedule.algorithm == Schedule.MANUAL_ORDER:
            for num, shift in enumerate(Shift.objects.filter(schedule=schedule).order_by('start')):
                shift.index = num
                shift.save(update_fields=['index'])

        priority_recalculate_shifts_for_service.delay_on_commit(schedule.service.id)
        return Response(status=200)

    def perform_destroy(self, schedule):
        full_recalculation_ids = []
        if schedule.is_no_order_consider():
            full_recalculation_ids = schedule.get_associated_schedules_id_list(include_self=False)
        Schedule.objects.filter(id=schedule.id).safe_delete()
        priority_recalculate_shifts_for_service.apply_async_on_commit(
            args=[schedule.service.id, full_recalculation_ids]
        )
        self.create_history_entry(schedule)

    @action(methods=['post'], detail=False)
    def check_changes(self, request):
        queryset = self.filter_queryset(self.get_queryset())
        checker = ScheduleFieldsChecker(request, self.serializer_class, queryset, self)
        checker.validate()
        data = checker.get_response_data()
        return Response(data=data, status=200)


class GapsView(NoPaginationListMixin, viewsets.ReadOnlyModelViewSet):
    queryset = Gap.objects.active()
    filter_class = GapFilterSet
    ordering = ('start',)
    pagination_class = None
    serializer_class = GapSerializer
    _permissions_to_proceed = 'view_duty'


class ShiftView(HistoryMixin, TvmAccessMixin, DontFilterOnGetObjectMixin, NoPaginationListMixin, viewsets.ModelViewSet):
    filter_class = ShiftFilterSet
    metadata_class = ShiftMetadata
    ordering = ('start',)
    permission_classes = [DutyShiftPermission]
    _permissions_to_proceed = 'view_duty'

    def get_serializer_class(self):
        if self.action in ('list', 'retrieve'):
            return ShiftReadSerializer

        if self.action in ('create', 'destroy'):
            return ShiftCreateSerializer

        return ShiftSerializer

    def get_queryset(self):
        if self.action == 'create':
            qs = Shift.objects.exclude(replace_for=None)
        elif self.action == 'destroy':
            qs = Shift.objects.filter(
                Q(replace_for__isnull=False)|
                Q(schedule__recalculate=False)
            )
        else:
            qs = Shift.objects.filter(replace_for=None)

        qs = qs.alive().select_related('staff', 'schedule', 'schedule__role__scope')

        if self.action in ('list', 'retrieve'):
            qs = qs.prefetch_related(
                'replaces',
                'replaces__staff',
                'replaces__schedule',
                'schedule__orders',
            )
        else:
            qs = qs.select_related('schedule__service')

        return qs

    def perform_destroy(self, instance):
        if instance.state == Shift.STARTED:
            try:
                instance.cancel()
            except IDMError:
                raise BadRequest('Wrong answer from IDM')
        super().perform_destroy(instance)


class AllowForDutyView(mixins.ListModelMixin, viewsets.GenericViewSet):
    _permissions_to_proceed = 'view_duty'

    serializer_class = AllowForDutyResponseSerializer

    def __init__(self, *args, **kwargs):
        super(AllowForDutyView, self).__init__(*args, **kwargs)
        self.schedule = None

    def get_serializer_context(self):
        context = super(AllowForDutyView, self).get_serializer_context()
        context['schedule'] = self.schedule
        return context

    def get_queryset(self):
        form = AllowForDutyRequestSerializer(data=self.request.query_params)
        form.is_valid(raise_exception=True)
        schedule = form.validated_data.get('schedule')
        self.schedule = schedule
        service = form.validated_data.get('service')
        roles = form.validated_data.get('role')
        ordered = form.validated_data.get('ordered')

        service_members = ServiceMember.objects.filter(service=service)
        params_q = Q()
        if not (roles or schedule):
            params_q = ~Q(
                role__code__in=Role.CAN_NOT_USE_FOR_DUTY | {Role.RESPONSIBLE_FOR_DUTY}
            )
        else:
            if roles:
                params_q |= Q(role__in=roles)
            if schedule:
                params_q |= schedule.get_role_q()
        service_members = service_members.filter(params_q)
        id_list = service_members.values_list('staff__id', flat=True)

        if schedule and ordered:
            future_shifts = (
                schedule.shifts.future()
                .fulltime()
                .filter(staff_id__in=id_list)
                .order_by('start')
                .select_related('staff')
            )[:len(id_list)]
            return list(OrderedDict.fromkeys(shift.staff for shift in future_shifts))
        else:
            return Staff.objects.filter(id__in=id_list)


class V4ScheduleView(TvmAccessMixin, DefaultFieldsMixin, ScheduleView):
    default_swagger_schema = SwaggerDuty
    TVM_ALLOWED_METHODS = {'GET', 'PATCH', 'DELETE'}

    default_fields = [
        'duration',
        'id',
        'name',
        'slug',
        'start_date',

        'service.id',
        'service.slug',
    ]
    serializer_class_upload_history = ShiftUploadHistorySerializer

    def get_serializer_class(self):
        if self.action in ('append', 'replace'):
            return self.serializer_class_upload_history

        return self.serializer_class

    def validate_upload_history(self, schedule, serializer):
        """
        Метод для валидации данных загрузки:
            * Проверяем имеет ли пользователь права на управления дежурствами.
            * Проверяем, что пытаемся загрузить историю только для одного графика.
        """

        serializer.is_valid(raise_exception=True)
        requester = self.request.person

        if not requester.can_modify_duty_in_service(schedule.service):
            raise PermissionDenied(f'You are not responsible for service {schedule.service.slug}')

        upload_schedules = {obj['schedule'] for obj in serializer.validated_data}
        if len(upload_schedules) > 1 or schedule not in upload_schedules:
            raise BadRequest(f'Wrong schedules {upload_schedules} for this calendar')

    def save_shifts_index(self, schedule):
        """
        Если график с порядком, то загруженные шифты должны быть проиндексированы.
        """

        if schedule.algorithm == Schedule.MANUAL_ORDER:
            for num, shift in enumerate(Shift.objects.filter(schedule=schedule).order_by('start')):
                shift.index = num
                shift.save(update_fields=['index'])

    def save_uploads_shifts(self, request, replace=False):
        schedule = self.get_object()
        serializer = self.serializer_class_upload_history(data=request.data, many=True)
        self.validate_upload_history(schedule, serializer)

        if replace:
            Shift.objects.filter(schedule=schedule).delete()

        serializer.save()
        self.save_shifts_index(schedule)

        priority_recalculate_shifts_for_service.delay_on_commit(schedule.service.id)

    @action(methods=['post'], detail=True, url_path='shifts/append', url_name='shifts-append')
    @transaction.atomic
    def append(self, request, pk=None):
        """
        Дозаливка смен:
            Не происходит затирания смен: ни прошлых, ни текущих, ни будущих.
        """

        self.save_uploads_shifts(request)
        return Response(status=200)

    @action(methods=['post'], detail=True, url_path='shifts/replace', url_name='shifts-replace')
    @transaction.atomic
    def replace(self, request, pk=None):
        """
        Полная загрузка истории:
            Удаляются все остальные смены.
        """

        self.save_uploads_shifts(request, replace=True)
        return Response(status=200)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['post'], detail=False)
    def check_changes(self, *args, **kwargs):
        return super(V4ScheduleView, self).check_changes(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    def list(self, *args, **kwargs):
        resp = super(V4ScheduleView, self).list(*args, **kwargs)
        ticket = getattr(self.request, 'tvm_service_ticket', None)
        page = self.request.GET.get('page')  # данные добавляем только на первую страницу
        cursor = self.request.GET.get('cursor')
        with_watcher = self.request.GET.get('with_watcher')
        if (
            (
                (ticket is not None and ticket.src in settings.WATCHER_PROXY_ENABLE_TVM_IDS)
                or with_watcher
            )
            and settings.WATCHER_PROXY_ENABLE
            and (not page or page == '1')
            and not cursor
        ):
            # добавим данные из новых дежурств
            watcher_client = WatcherClient()
            service_id = self.request.GET.get('service')
            schedule_id = self.request.GET.get('id')
            service_slug = self.request.GET.get('service__slug')
            schedules = watcher_client.get_schedules(
                service_id=service_id,
                service_slug=service_slug,
                state='active',
                schedule_id=schedule_id,
                field_params=['show_in_staff'],
            )
            services = {
                service.id: service for service in
                Service.objects.filter(pk__in=[schedule['service_id'] for schedule in schedules])
            }
            for schedule in schedules:
                service = services[schedule['service_id']]
                resp.data['results'].append(
                    {
                        'id': schedule['id'],
                        'slug': schedule['slug'],
                        'name': schedule['name'],
                        'role': None,
                        'is_important': False,
                        'from_watcher': True,
                        'show_in_staff': schedule['show_in_staff'],
                        'service': {
                            'id': service.id,
                            'slug': service.slug,
                            'name': {
                                'en': service.name_en,
                                'ru': service.name,
                            }
                        }
                    }
                )
            if 'count' in resp.data:
                resp.data['count'] += len(schedules)

        return resp

    def update(self, *args, **kwargs):
        """
        Изменения одного из полей duration , only_workdays , start_date , algorithm  приводят к полному пересчету графика,
        при котором сохраняется только активная смена дежурных.
        """
        return super(V4ScheduleView, self).update(*args, **kwargs)


class V4GapsView(DefaultFieldsMixin, GapsView):
    default_fields = [
        'end',
        'full_day',
        'id',
        'start',

        'person.login',
        'person.uid',
    ]


class V4ShiftView(DefaultFieldsMixin, ShiftView):
    default_swagger_schema = SwaggerDuty

    default_fields = [
        'id',
        'is_approved',
        'start',
        'start_datetime',
        'end',
        'end_datetime',

        'person.uid',
        'person.login',

        'schedule.id',
        'schedule.name',
    ]

    def list(self, *args, **kwargs):
        """
        Активные смены сервиса можно получить через /api/v4/duty/on_duty/
        """
        resp = super(V4ShiftView, self).list(*args, **kwargs)
        ticket = getattr(self.request, 'tvm_service_ticket', None)
        with_watcher = self.request.GET.get('with_watcher')
        if (
            (
                (ticket is not None and ticket.src in settings.WATCHER_PROXY_ENABLE_TVM_IDS)
                or with_watcher
            )
            and settings.WATCHER_PROXY_ENABLE
        ):
            #  пришел стафф, нужно добавить данные из новых дежурств
            watcher_client = WatcherClient()
            service_id = self.request.GET.get('service')
            schedule_id = self.request.GET.get('schedule')
            schedule_slug = self.request.GET.get('schedule__slug')
            service_slug = self.request.GET.get('service__slug')
            date_from = self.request.GET.get('date_from')
            date_to = self.request.GET.get('date_to')

            shifts = watcher_client.get_shifts(
                service_id=service_id,
                schedule_id=schedule_id,
                service_slug=service_slug,
                schedule_slug=schedule_slug,
                date_to=date_to,
                date_from=date_from,
            )
            _add_watcher_shifts_data(
                watcher_shifts=shifts,
                result=resp.data['results'],
            )
        return resp

    def create(self, *args, **kwargs):
        """
        Создавать можно только замены к смене, поэтому параметр ``replace_for`` обязательный.
        """
        return super(V4ShiftView, self).create(*args, **kwargs)

    def update(self, *args, **kwargs):
        """
        Можно заменить дежурного, подтвердить дежурство или отредактировать список замен (заменить или удалить).
        Другие поля меняться не будут.
        """
        return super(V4ShiftView, self).update(*args, **kwargs)

    def destroy(self, *args, **kwargs):
        """
        Используется для удаления замены или смены в расписании с флагом: recalculate = false.
        Обычную смену удалить нельзя
        """
        return super(V4ShiftView, self).destroy(*args, **kwargs)


class V4AllowForDutyView(DefaultFieldsMixin, AllowForDutyView, TvmAccessMixin):
    """
    Возвращает список всех доступных Дежурных  для сервиса.
    """
    default_swagger_schema = SwaggerDuty

    default_fields = ['id', 'login', 'order', 'start_with', 'uid']
    permission_classes = [DutyShiftPermission]


class V4OnDutyView(DefaultFieldsMixin,
                   mixins.ListModelMixin, viewsets.GenericViewSet,
                   TvmAccessMixin,
                   SelectOnlyFieldsMixin,
                   OrderingMixin):
    """
    Возвращает объекты Shift , активные на данный момент
    """
    default_swagger_schema = SwaggerDuty

    pagination_class = None
    filter_class = ShiftBaseFilterSet
    serializer_class = ShiftReadSerializer
    permission_classes = [DutyShiftPermission]
    _permissions_to_proceed = 'view_duty'
    ordering_fields = ('index', 'id')
    default_fields = [
        'id',
        'is_approved',
        'start',
        'start_datetime',
        'end',
        'end_datetime',

        'person.uid',
        'person.login',
        'person.first_name',
        'person.last_name',
        'person.telegram_account',

        'schedule.id',
        'schedule.name',
        'schedule.slug',
    ]

    def get_queryset(self):
        now = timezone.now()
        qs = (
            Shift.objects
            .filter(
                schedule__service__isnull=False,
                start_datetime__lte=now,
                end_datetime__gt=now,
                state__in=[Shift.STARTED, Shift.SCHEDULED],
            ).alive().select_related('schedule', 'replace_for')
        )

        # True - праздник
        # False - выходной
        # None - нет объекта, тк это рабочий
        is_holiday = date_is_holiday_type(utils.today())

        if is_holiday:
            # праздник
            qs = qs.filter(schedule__duty_on_holidays=True)
        elif is_holiday is False:
            # выходной
            qs = qs.filter(schedule__duty_on_weekends=True)

        return qs.remove_replaced_shifts()

    def filter_queryset(self, queryset):
        queryset = super().filter_queryset(queryset)
        page_size = ABCPagination().get_page_size(self.request)
        return queryset[:page_size]

    def list(self, request, *args, **kwargs):
        # для обратной совместимости с ручкой services/on_duty,
        # чтобы редиректом никому ничего не поломать
        results = super(V4OnDutyView, self).list(request, *args, **kwargs).data

        ticket = getattr(self.request, 'tvm_service_ticket', None)
        with_watcher = self.request.GET.get('with_watcher')
        service_id = self.request.GET.get('service')
        schedule_id = self.request.GET.get('schedule')
        schedule_slug = self.request.GET.get('schedule__slug')
        service_slug = self.request.GET.get('service__slug')

        proxy_from_magiclinks = False
        if (
            not results and
            hasattr(request.user, 'username') and
            request.user.username == settings.MAGICLINKS_ROBOT_LOGIN
        ):
            duty_to_watcher = DutyToWatcher.objects.filter(abc_id=schedule_id).first()
            if duty_to_watcher:
                schedule_id = str(duty_to_watcher.watcher_id)
                proxy_from_magiclinks = True
        if (
            (
                (ticket is not None and ticket.src in settings.WATCHER_PROXY_ENABLE_TVM_IDS)
                or with_watcher or proxy_from_magiclinks
            )
            and settings.WATCHER_PROXY_ENABLE
        ):
            watcher_client = WatcherClient()

            shifts = watcher_client.get_shifts(
                service_id=service_id,
                schedule_id=schedule_id,
                service_slug=service_slug,
                schedule_slug=schedule_slug,
                current=True,
            )
            _add_watcher_shifts_data(
                watcher_shifts=shifts,
                result=results,
            )

        return Response(results)


class FrontendScheduleView(ScheduleView, TvmAccessMixin):
    default_swagger_schema = SwaggerFrontend
    serializer_class = FrontendScheduleSerializer

    def get_queryset(self):
        queryset = super(FrontendScheduleView, self).get_queryset()

        service_member_base_q = Q(service_id=OuterRef('service_id'), staff=self.request.person.staff)
        service_member_role_q = service_member_base_q & Q(role_id=OuterRef('role_id'))
        service_member_norole_q = service_member_base_q & (
            ~Q(role__code__in=Role.CAN_NOT_USE_FOR_DUTY) | Q(role__code__isnull=True)
        )

        requester_is_role_member = Exists(ServiceMember.objects.filter(service_member_role_q))
        requester_is_norole_member = Exists(ServiceMember.objects.filter(service_member_norole_q))
        order_exists = Exists(Order.objects.filter(schedule=OuterRef('pk'), staff=self.request.person.staff))

        requester_is_member_q = (
            Q(role__isnull=True, requester_is_norole_member=True) |
            Q(role__isnull=False, requester_is_role_member=True)
        )

        requester_in_duty = Case(
            When(requester_is_member_q & (Q(algorithm=Schedule.NO_ORDER) | Q(order_exists=True)), then=True),
            default=False,
            output_field=BooleanField()
        )

        return queryset.annotate(
            requester_is_role_member=requester_is_role_member,
            requester_is_norole_member=requester_is_norole_member,
            order_exists=order_exists
        ).annotate(
            requester_in_duty=requester_in_duty
        )


class V4ScheduleCursorView(V4ScheduleView):
    """
    В документации нужна только ручка list
    """
    default_swagger_schema = None

    pagination_class = ABCCursorPagination
    ordering_fields = ('id',)
    ordering = ('-id',)
    default_cursor_ordering = ('-id',)

    @swagger_auto_schema(auto_schema=SwaggerDuty)
    def list(self, *args, **kwargs):
        """
        Возвращает объекты Schedule.
        """
        return super(V4ScheduleCursorView, self).list(*args, **kwargs)


class V4DutyToWatcherView(NoPaginationListMixin, DefaultFieldsMixin, viewsets.ModelViewSet):
    TVM_ALLOWED_METHODS = {'POST'}

    queryset = DutyToWatcher.objects.all()
    pagination_class = None
    permission_classes = (DutyToWatcherPermission,)
    serializer_class = DutyToWatcherSerializer
    _permissions_to_proceed = 'view_duty'
