import logging
import datetime

from django.conf import settings
from django.contrib.postgres.fields import ArrayField
from django.core.validators import MinValueValidator
from django.db import models, transaction
from django.db.models import Q, F, Subquery, OuterRef, Exists
from django.db.models.functions import Coalesce
from django.utils import timezone
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from startrek_client.exceptions import StartrekError
from django_fsm import FSMField, transition

from plan import exceptions
from plan.duty.tracker import set_component_lead, remove_component_lead
from plan.api.exceptions import ValidationError
from plan.api.idm import actions
from plan.common.models import TimestampedModel
from plan.common.utils import timezone as utils
from plan.common.utils.dates import datetime_from_str
from plan.common.utils.timezone import make_localized_datetime
from plan.duty.exceptions import NoStartedShiftsException
from plan.holidays.utils import trim_holidays_and_weekends, end_datetime_to_end_date
from plan.roles.models import Role
from plan.services.models import Service, ServiceMember
from plan.staff.models import Staff


logger = logging.getLogger(__name__)


def get_days_for_begin_shift_notification_default():
    return [0, 1, 7]


class GapQuerySet(models.QuerySet):
    def soft_delete(self):
        self.update(status=Gap.DELETED, deleted_at=timezone.now())

    def active(self):
        return self.filter(status=Gap.ACTIVE)


class Gap(models.Model):
    objects = GapQuerySet.as_manager()

    ACTIVE = 'active'
    DELETED = 'deleted'
    STATUSES = (
        (ACTIVE, _('Активный')),
        (DELETED, _('Удаленный')),
    )

    status = models.CharField(choices=STATUSES, default=ACTIVE, max_length=20, verbose_name=_('Статус отсутствия'))
    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_('Время создания'))
    deleted_at = models.DateTimeField(blank=True, null=True, verbose_name=_('Время удаления'))

    gap_id = models.BigIntegerField(verbose_name=_('Id в Гэпе'), db_index=True, unique=True)
    staff = models.ForeignKey(Staff, related_name='gaps')
    type = models.TextField(verbose_name=_('Тип отсутствия'))
    start = models.DateTimeField(verbose_name=_('Начало'), db_index=True)
    end = models.DateTimeField(verbose_name=_('Конец'), db_index=True)
    full_day = models.BooleanField(verbose_name=_('Отсутствие на весь день'))
    work_in_absence = models.BooleanField(verbose_name=_('Будет работать во время отсутствия'))

    @classmethod
    def from_gap_api_data(cls, data, staff_pk_cache=None):
        if staff_pk_cache is not None:
            staff_pk = staff_pk_cache[data['person_login']]
        else:
            staff_pk = Staff.objects.get(login=data['person_login']).id

        return cls(
            gap_id=data['id'],
            staff_id=staff_pk,
            start=datetime_from_str(data['date_from']).replace(tzinfo=timezone.utc),
            end=datetime_from_str(data['date_to']).replace(tzinfo=timezone.utc),
            work_in_absence=data['work_in_absence'],
            full_day=data['full_day'],
            type=data['workflow'],
            created_at=timezone.now(),
        )


class ScheduleQuerySet(models.QuerySet):

    def safe_delete(self):
        from plan.duty.tasks import remove_inactive_schedules

        self.update(status=Schedule.DELETED, deleted_at=timezone.now())
        remove_inactive_schedules.apply_async()

    def no_order_consider(self):
        return self.filter(consider_other_schedules=True, algorithm=Schedule.NO_ORDER)

    def associated_with(self, schedule):
        return self.filter(~models.Q(id=schedule.id), service=schedule.service, status=Schedule.ACTIVE)

    def for_role(self, role):
        return self.filter(Q(role=role) | Q(role=None))

    def no_order(self):
        return self.filter(algorithm=Schedule.NO_ORDER)

    def manual_order(self):
        return self.filter(algorithm=Schedule.MANUAL_ORDER)

    def active(self):
        return self.filter(service__state__in=Service.states.ACTIVE_STATES, status=Schedule.ACTIVE)

    def deleted(self):
        return self.filter(status=Schedule.DELETED)

    def with_set_days_for_begin_notifications(self):
        return self.active().filter(days_for_begin_shift_notification__isnull=False)

    def with_set_days_for_problem_notification(self):
        return self.active().filter(days_for_problem_notification__isnull=False)

    def with_recalculation(self):
        return self.filter(recalculate=True)


class ScheduleManager(models.Manager.from_queryset(ScheduleQuerySet)):

    def add_or_update_schedule(self, service, schedule_data, pk=None):
        algorithm = schedule_data.get('algorithm', None)
        only_workdays = schedule_data.get('only_workdays', None)

        if only_workdays is not None:
            schedule_data['duty_on_holidays'] = not only_workdays
            schedule_data['duty_on_weekends'] = not only_workdays

        orders = schedule_data.pop('orders', None)
        start_with = schedule_data.pop('start_with', None)

        if pk is not None:
            schedule = Schedule.objects.filter(pk=pk)

            if not schedule.exists():
                raise ValidationError('Wrong schedule.id %s' % pk)

            schedule.update(**schedule_data)
            schedule = schedule.first()
            algorithm = schedule.algorithm if algorithm is None else algorithm

        else:
            schedule_data['service'] = service
            schedule = self.create(**schedule_data)

        if algorithm != Schedule.MANUAL_ORDER or orders:
            schedule.orders.all().delete()

        if orders or start_with:
            if orders is not None:
                Order.objects.add_orders(orders, schedule)

            else:
                orders = schedule.orders.order_by('order')

            if start_with is None:
                start_with = orders[0]

            schedule.save_start_with_as_offset(start_with)

        return schedule


class Schedule(models.Model):
    objects = ScheduleManager()

    MANUAL_ORDER = 'manual_order'
    NO_ORDER = 'no_order'
    ALGORITHM = (
        (MANUAL_ORDER, _('Порядок дежурных задается вручную')),
        (NO_ORDER, _('Автоматичекий выбор следующего дежурного')),
    )
    DEFAULT_ALGORITHM = NO_ORDER

    DEFAULT_DUTY_START_TIME = datetime.time(hour=0)

    ACTIVE = 'active'
    DELETED = 'deleted'
    STATUSES = (
        (ACTIVE, _('Активный')),
        (DELETED, _('Удаленный')),
    )

    name = models.TextField(verbose_name=_('Название'))
    description = models.TextField(verbose_name=_('Описание'), blank=True, null=True, default='')
    persons_count = models.PositiveSmallIntegerField(
        default=1,
        validators=[MinValueValidator(1)],
        verbose_name=_('Количество дежурных')
    )
    role = models.ForeignKey(Role, null=True, blank=True, related_name='+', verbose_name=_('Роль'))
    consider_other_schedules = models.BooleanField(default=True, verbose_name=_('Учитывать другие дежурства'))
    algorithm = models.CharField(choices=ALGORITHM, default=DEFAULT_ALGORITHM, max_length=20,
                                 verbose_name=_('Алгоритм выбора дежурного'), db_index=True)
    manual_ordering_offset = models.IntegerField(default=0, verbose_name=_('Сдвиг при расчете дежурного'))

    service = models.ForeignKey(Service, verbose_name=_('Сервис'), related_name='schedules', null=True, db_index=True)
    duration = models.DurationField(verbose_name=_('Длительность дежурства'))
    only_workdays = models.BooleanField(default=False, verbose_name=_('Только рабочие дни'))
    # для праздников и выходных дефолт аналогичен текущему only_workdays=False => в новых True
    duty_on_weekends = models.BooleanField(default=True, verbose_name=_('Дежурить по выходным'))
    duty_on_holidays = models.BooleanField(default=True, verbose_name=_('Дежурить по праздникам'))
    start_date = models.DateField(verbose_name=_('Дата начала'), help_text="Например: '2020-07-01'")
    start_time = models.TimeField(verbose_name=_('Время начала по Москве'), default=DEFAULT_DUTY_START_TIME)
    status = models.CharField(choices=STATUSES, default=ACTIVE, max_length=20)
    allow_sequential_shifts = models.BooleanField(
        default=False,
        verbose_name=_('Позволять одному человеку дежурить несколько смен подряд'),
    )

    deleted_at = models.DateTimeField(null=True, blank=True, verbose_name=_('Время удаления'))

    autoapprove_timedelta = models.DurationField(
        default=settings.DEFAULT_AUTOAPPROVE_TIMEDELTA,
        verbose_name=_('За сколько времени до начала смены автоматически подтверждать смену'),
    )

    role_on_duty = models.ForeignKey(
        Role,
        null=True,
        blank=True,
        related_name='schedules',
        verbose_name=_('Роль на время дежурства')
    )

    slug = models.SlugField(
        verbose_name=_('Код'),
        blank=True,
    )

    days_for_problem_notification = models.PositiveIntegerField(
        null=True,
        blank=True,
        default=14,
        verbose_name=_('За сколько дней до начала дежурства уведомлять о проблемах'),
    )

    days_for_begin_shift_notification = ArrayField(
        models.PositiveIntegerField(),
        blank=True,
        null=True,
        default=get_days_for_begin_shift_notification_default,
        verbose_name=_('За сколько дней до начала дежурства отправлять уведомления'),
    )

    show_in_staff = models.BooleanField(
        default=True,
        verbose_name=_('Показывать как отсутствие на Staff')
    )

    recalculate = models.BooleanField(
        default=True,
        verbose_name=_('Пересчитывать смены'),
    )

    tracker_queue_id = models.PositiveIntegerField(blank=True, null=True)
    tracker_component_id = models.PositiveIntegerField(blank=True, null=True)
    is_important = models.BooleanField(default=False)

    class Meta:
        unique_together = [('service', 'name'), ('service', 'slug')]

    def __str__(self):
        return 'Duty Schedule %s of %s service' % (self.name, self.service)

    def get_role_q(self):
        if self.role is None:
            return ~Q(role__code__in=Role.EXCLUDE_FROM_DUTY) | Q(role__code__isnull=True)

        return Q(role=self.role)

    def get_associated_schedules_id_list(self, include_self=True):
        result = list(Schedule.objects.associated_with(self).values_list('id', flat=True))
        if include_self:
            result.append(self.id)
        return result

    def save_start_with_as_offset(self, start_with):
        from plan.duty.schedulers import ManualOrderingScheduler
        offset = ManualOrderingScheduler.get_offset(self, start_with)
        self.manual_ordering_offset = offset
        self.save(update_fields=['manual_ordering_offset'])

    def is_no_order_consider(self):
        return self.algorithm == Schedule.NO_ORDER and self.consider_other_schedules

    def get_role_on_duty(self):
        return self.role_on_duty if self.role_on_duty is not None else Role.objects.globalwide().get(code=Role.DUTY)

    def get_next_shift_after_active(self):
        last_active = (
            self.shifts.filter(replace_for=None)
            .current_shifts()
            .order_by('start')
            .last()
        )
        if last_active:
            return (
                self.shifts.filter(start__gte=last_active.end, replace_for=None)
                .order_by('start', 'index')
                .first()
            )

    def get_start_with(self):
        shift = self.get_next_shift_after_active()
        if shift:
            orders = list(self.orders.order_by('order'))
            if orders:
                return orders[(int(shift.index) + self.manual_ordering_offset) % len(orders)].staff

    def get_staff_order(self, staff):
        return self.orders.filter(staff=staff).values_list('order', flat=True).first()

    def is_staff_active_duty(self, staff):
        return self.shifts.filter(staff=staff).current_shifts().fulltime().exists()

    def account_for_holidays(self, dt: datetime.datetime) -> datetime.datetime:
        if not self.duty_on_holidays or not self.duty_on_weekends:
            return trim_holidays_and_weekends(dt, self.duty_on_holidays, self.duty_on_weekends)
        else:
            return dt

    @transaction.atomic
    def update_shifts_date(self):
        current_shifts = self.shifts.current_shifts()

        for shift in current_shifts:
            shift.end_datetime = self.account_for_holidays(
                make_localized_datetime(
                    shift.end_datetime.astimezone(settings.DEFAULT_TIMEZONE).date(), self.start_time
                )
            )
            shift.end = end_datetime_to_end_date(shift.end_datetime)

            shift.save(update_fields=['end', 'end_datetime'])

        future_shifts = self.shifts.future().scheduled()
        for shift in future_shifts:
            shift.start_datetime = make_localized_datetime(
                shift.start_datetime.astimezone(settings.DEFAULT_TIMEZONE).date(), self.start_time
            )
            shift.start = shift.start_datetime.astimezone(settings.DEFAULT_TIMEZONE).date()

            shift.end_datetime = self.account_for_holidays(
                make_localized_datetime(
                    shift.end_datetime.astimezone(settings.DEFAULT_TIMEZONE).date(), self.start_time
                )
            )
            shift.end = end_datetime_to_end_date(shift.end_datetime)

            shift.save(update_fields=['start', 'start_datetime', 'end', 'end_datetime'])


class OrderManager(models.Manager):

    def add_orders(self, orders, schedule):
        role_q = schedule.get_role_q()
        service = schedule.service
        possible_login = service.members.filter(role_q).values_list('staff__login', flat=True)

        orders_data = []
        for order, staff in enumerate(orders):
            if staff.login not in possible_login:
                raise ValidationError(
                    '%s не может быть дежурным в теге %s, т.к. не имеет подходящей роли в сервисе %s'
                    % (staff.login, schedule.name, service.slug))

            orders_data.append(
                Order(
                    schedule=schedule,
                    order=order,
                    staff=staff,
                )
            )

        self.bulk_create(orders_data)


class Order(models.Model):
    objects = OrderManager()

    schedule = models.ForeignKey(Schedule, related_name='orders', null=False)
    staff = models.ForeignKey(
        Staff,
        related_name='orders',
        verbose_name=_('Пользователь'),
        null=True,
        blank=True,
    )
    order = models.IntegerField(null=False)

    class Meta:
        unique_together = ('schedule', 'staff')


class ShiftQuerySet(models.QuerySet):

    def started(self):
        return self.filter(state=Shift.STARTED)

    def scheduled(self):
        return self.filter(state=Shift.SCHEDULED)

    def not_started(self):
        return self.filter(~models.Q(state=Shift.STARTED))

    def future(self):
        return self.filter(start_datetime__gt=utils.now())

    def current_shifts(self):
        today = utils.now()
        return self.filter(start_datetime__lte=today, end_datetime__gte=today)

    def future_and_present(self):
        return self.filter(end_datetime__gte=utils.today())

    def past_shifts(self):
        return self.filter(end_datetime__lte=utils.now())

    def with_staff(self):
        return self.filter(staff__isnull=False)

    def problematic(self):
        return self.filter(has_problems=True)

    def fulltime(self):
        return self.filter(replace_for=None)

    def parttime(self):
        return self.filter(replace_for__isnull=False)

    def alive(self):
        return self.filter(schedule__service__state__in=Service.states.ACTIVE_STATES,
                           schedule__status=Schedule.ACTIVE)

    def remove_replaced_shifts(self):
        qs = self.exclude(pk__in=self.parttime().values_list('replace_for', flat=True))
        return qs

    def for_notify_about_begin(self, day, notification_id):
        today = utils.today()
        return (
            self.alive().with_staff()
            .exclude(
                Q(notifications__recipient=F('staff'), notifications__notification_id=notification_id)
                & (
                    Q(notifications__days_before_shift=(F('start') - day)) |
                    Q(notifications__real_days_before_shift=(F('start') - today))
                )
            )
        )

    def startable(self):
        filter_time = utils.now() + settings.SHIFT_BEGIN_BEFORE_DUTY_START_TIMEDELTA
        starting_q = Q(
            state=Shift.SCHEDULED,
            start_datetime__lte=filter_time,
            end_datetime__gt=filter_time,
        )
        started_without_member_q = Q(
            state=Shift.STARTED,
            member_id=None,
        )
        return self.with_staff().annotate_member_id().filter(starting_q | started_without_member_q)

    def annotate_member_id(self):
        member_id_q = ServiceMember.objects.filter(
            role_id=OuterRef('cached_role_id'),
            staff=OuterRef('staff'),
            service=OuterRef('schedule__service'),
            from_department=None,
        )[:1]
        return (
            self
            .annotate(cached_role_id=Coalesce(
                Subquery(Schedule.objects.filter(shifts__id=OuterRef('pk')).values('role_on_duty')[:1]),
                Subquery(Role.objects.globalwide().filter(code__in=[Role.DUTY]).values('pk')[:1])
            ))
            .annotate(member_id=Subquery(member_id_q.values('id')))
        )

    def finishable(self):
        return (
            self
            .started()
            .filter(end_datetime__lt=utils.now() - settings.SHIFT_FINISH_AFTER_DUTY_START_TIMEDELTA)
        )

    def annotate_intersecting(self):
        now = timezone.now()
        intersections_q = Shift.objects.filter(
            schedule__service_id=OuterRef('schedule__service_id'),
            staff_id=OuterRef('staff_id'),
            state=Shift.STARTED,
            start__lte=now,
            end__gte=now,
            role=OuterRef('role'),
        ).exclude(pk=OuterRef('pk'))
        return self.annotate(intersecting=Exists(intersections_q))


class Shift(models.Model):
    objects = ShiftQuerySet.as_manager()

    SCHEDULED = 'scheduled'
    STARTED = 'started'
    FINISHED = 'finished'
    CANCELLED = 'cancelled'
    STATES = (
        (SCHEDULED, _('Запланировано')),
        (STARTED, _('Начато')),
        (FINISHED, _('Завершено')),
        (CANCELLED, _('Отменено')),
    )

    staff = models.ForeignKey(
        Staff,
        null=True,
        related_name='duties',
        verbose_name=_('Пользователь')
    )

    role = models.ForeignKey(
        Role,
        models.SET_NULL,
        null=True,
        blank=True,
        verbose_name=_('Роль'),
    )

    schedule = models.ForeignKey(Schedule, related_name='shifts', verbose_name=_('График дежурств'), db_index=True)
    start = models.DateField(verbose_name=_('Начало'), db_index=True)
    end = models.DateField(verbose_name=_('Конец'), db_index=True)
    start_datetime = models.DateTimeField(
        verbose_name=_('Начало (+время)'),
        db_index=True,
        null=True,
        help_text="Формат '2021-01-01T13:00:00'"
    )
    end_datetime = models.DateTimeField(
        verbose_name=_('Конец (+время)'),
        db_index=True,
        null=True,
        help_text="Формат '2021-01-18T19:00:00'"
    )
    has_problems = models.BooleanField(default=False, verbose_name=_('Есть проблемы'))
    is_approved = models.BooleanField(default=False, verbose_name=_('Подтверждено руководителем'))
    approved_by = models.ForeignKey(Staff, null=True, blank=True, verbose_name=_('Кем подтверждено'))
    approve_datetime = models.DateTimeField(null=True, blank=True, verbose_name=_('Время подтверждения'))
    replace_for = models.ForeignKey(
        'self',
        related_name='replaces',
        null=True,
        blank=True,
        verbose_name=_('Замены'),
        on_delete=models.CASCADE,
        db_index=True,
    )

    state = FSMField(verbose_name=_('Статус'), choices=STATES, default=SCHEDULED, db_index=True)

    index = models.IntegerField(null=True, blank=True, verbose_name=_('Индекс'))

    created_at = models.DateTimeField(auto_now_add=True, null=True, verbose_name=_('Создание смены'),)
    updated_at = models.DateTimeField(auto_now=True, null=True, verbose_name=_('Изменение смены'),)

    class Meta:
        verbose_name = _('Смена')
        verbose_name_plural = _('Смены')
        ordering = ('id',)

    def get_member(self):
        return ServiceMember.objects.filter(
            service=self.schedule.service,
            staff=self.staff,
            role=self.role,
        ).first()

    @property
    def service(self):
        return self.schedule.service

    @cached_property
    def localize_start_datetime(self):
        # Для использования этого property в цикле нужно заранее выбрать из базы schedule
        return make_localized_datetime(self.start, self.schedule.start_time)

    @cached_property
    def localize_end_datetime(self):
        # Для использования этого property в цикле нужно заранее выбрать из базы schedule
        # Из-за добавления времени начала/окончания дежурства, текущая смена фактически завершается не в end,
        # а на следующий день в start_time, поэтому нам нужно прибавить к дате 1 день
        return make_localized_datetime(self.end, self.schedule.start_time) + timezone.timedelta(days=1)

    def _has_intersection_with_other_shifts(self):
        now = timezone.now()
        return (
            Shift.objects
            .filter(
                schedule__service=self.schedule.service,
                staff=self.staff,
                state=Shift.STARTED,
                start__lte=now,
                end__gte=now,
                role=self.role,
            )
            .exclude(pk=self.pk)
        ).exists()

    def deprive_role(self, comment=None):
        member = self.get_member()
        if member is not None and member.autorequested:
            has_intersection = self._has_intersection_with_other_shifts()
            logger.debug(
                'Depriving role. shift:%s, intersection:%s, member:%s',
                self.id,
                has_intersection,
                member
            )
            if not has_intersection:
                actions.deprive_role(member, comment=comment)

    @transition(field=state, source=STARTED, target=CANCELLED)
    def cancel(self):
        self.deprive_role(comment='Отмена дежурства')
        self.remove_component_lead()
        if self.replace_for_id and self.replace_for.state == Shift.STARTED:
            self.replace_for.set_component_lead()

    def set_component_lead(self):
        if self.schedule.tracker_component_id and self.staff_id:
            try:
                set_component_lead(
                    component_id=self.schedule.tracker_component_id,
                    staff=self.staff,
                )
            except StartrekError as exc:
                logger.warning(
                    f'Exception appear while setting component lead: {repr(exc)}, shift: {self.id}'
                )

    def remove_component_lead(self):
        if self.schedule.tracker_component_id and self.staff_id:
            try:
                remove_component_lead(
                    component_id=self.schedule.tracker_component_id,
                    staff=self.staff,
                )
            except StartrekError as exc:
                logger.warning(
                    f'Exception appear while removing component lead: {repr(exc)}, shift: {self.id}'
                )

    def find_or_create_role(self):
        role = self.role if self.role is not None else self.schedule.get_role_on_duty()
        service = self.service
        try:
            ServiceMember.objects.get(staff=self.staff, service=service, role=role, from_department=None)
        except ServiceMember.DoesNotExist:
            actions.request_membership(service, self.staff, role, comment='Начало дежурства', silent=True)

    @transition(field=state, source=(SCHEDULED, CANCELLED), target=STARTED)
    def begin(self):
        self.find_or_create_role()
        self.set_component_lead()

    @transition(field=state, source=STARTED, target=FINISHED)
    def finish(self):
        if not self.schedule.shifts.filter(state=self.STARTED).exclude(pk=self.pk).exists():
            raise NoStartedShiftsException()

        self.deprive_role(comment='Конец дежурства')

    @transition(field=state, source='*', target=FINISHED)
    def forced_finish(self):
        pass

    def check_incorrect_replaces_intervals(self, replaces):
        if not replaces:
            return
        replaces.sort(key=lambda shift: shift.start_datetime)
        if replaces[0].start_datetime < self.start_datetime or replaces[-1].end_datetime > self.end_datetime:
            return True
        for index in range(len(replaces) - 1):
            if replaces[index].end_datetime > replaces[index + 1].start_datetime:
                return True

    def update_replaces(self, new_replaces, requester=None):
        """
        :param new_replaces: Список json-ок шифтов.
        Если id шифта-замены указан - обновляем изменившиеся поля, если нет - создаем новый
        """
        old_replaces = {shift.id: shift for shift in self.replaces.all()}
        id_to_save = set()
        replaces_to_create = []
        for shift in new_replaces:
            if 'id' not in shift:
                replaces_to_create.append(shift)
            else:
                shift_id = shift.pop('id')
                try:
                    old_shift = old_replaces[shift_id]
                except KeyError:
                    raise exceptions.NotFoundError(message='Shift %s not found' % shift_id)
                changed_fields = []
                for field, value in shift.items():
                    if getattr(old_shift, field) != value:
                        setattr(old_shift, field, value)
                        changed_fields.append(field)
                if changed_fields:
                    old_shift.save(update_fields=changed_fields)
                id_to_save.add(old_shift.id)
        replaces_to_create = [
            Shift(**dict(
                {
                    'replace_for': self,
                    'schedule': self.schedule,
                    'is_approved': True,
                    'approved_by': requester,
                }, **replace
            ))
            for replace in replaces_to_create
        ]
        if self.check_incorrect_replaces_intervals(
            replaces_to_create + [shift for shift in old_replaces.values() if shift.id in id_to_save]
        ) or any([shift.end_datetime <= shift.start_datetime for shift in replaces_to_create]):
            raise exceptions.DataValidationError(message='Replace date incorrect')
        for replace_id, replace in old_replaces.items():
            if replace_id not in id_to_save:
                if replace.state == Shift.STARTED:
                    replace.cancel()
                replace.delete()
        Shift.objects.bulk_create(replaces_to_create)

    def has_equal_problems(self, other_shift):
        """
        Сравнивает проблемы шифтов.
            * Если есть проблемные замены, значит проблемы точно не равны.
            * Если причин больше одной и стафы не совпадают => проблемы не равны
        """

        if Shift.objects.problematic().parttime().filter(replace_for__in=[self, other_shift]).exists():
            return False

        reasons_problems = set(
            Problem.objects.active().filter(shift__in=[self, other_shift])
            .values_list('reason', flat=True)
        )
        return len(reasons_problems) <= 1 and self.staff == other_shift.staff

    def __str__(self):
        return 'Duty Shift %s from %s to %s of %s schedule' % (self.staff, self.start, self.end, self.schedule)

    def days_before_shift(self, days_list) -> (int, int):
        real_days_before_shift = (self.start - utils.today()).days
        setting_days_before_shift = None

        for days_count in days_list:
            if real_days_before_shift < days_count:
                break
            setting_days_before_shift = days_count

        return setting_days_before_shift, real_days_before_shift

    def set_approved(self, is_approved, approved_by, save=True):
        self.is_approved = is_approved
        self.approved_by = approved_by if is_approved else None
        self.approve_datetime = timezone.now() if is_approved else None
        if save:
            self.save(update_fields=['is_approved', 'approved_by', 'approve_datetime'])


class ProblemQuerySet(models.QuerySet):

    def new(self, reason=None):
        if reason:
            return self.filter(status=Problem.NEW, reason=reason)
        else:
            return self.filter(status=Problem.NEW)

    def active(self, reason=None):
        queryset = self.filter(
            Q(schedule__service__state__in=Service.states.ACTIVE_STATES) |
            Q(shift__schedule__service__state__in=Service.states.ACTIVE_STATES)
        )
        if reason:
            return queryset.filter(reason=reason).exclude(status=Problem.RESOLVED)
        else:
            return queryset.exclude(status=Problem.RESOLVED)

    def of_shifts(self):
        return self.filter(shift__isnull=False)

    def only_nearest(self):
        return self.filter(
            shift__schedule__days_for_problem_notification__isnull=False,
            datetime__lte=utils.today() +
            datetime.timedelta(days=1) * F('shift__schedule__days_for_problem_notification')
        )

    def of_schedules(self):
        return self.filter(schedule__isnull=False)

    def with_enabled_notifications(self):
        return self.filter(schedule__days_for_problem_notification__isnull=False)

    def of_service(self, service):
        return self.filter(Q(shift__schedule__service=service) | Q(schedule__service=service))

    def ready_to_notification(self):
        threshold = timezone.now() - settings.PROBLEM_NOTIFICATION_SILENT_TIME
        queryset = self.filter(Q(report_date__isnull=True) | Q(report_date__lt=threshold))
        # Дополнительная проверка
        return queryset.exclude(notifications__created_at__gte=threshold)

    def future(self):
        return self.filter(shift__end__gt=utils.today())

    def future_and_present(self):
        return self.filter(shift__end_datetime__gte=timezone.now())

    def set_reported(self):
        return self.update(status=Problem.REPORTED, report_date=timezone.now())

    def set_resolved(self):
        return self.update(status=Problem.RESOLVED, resolve_date=timezone.now())


class Problem(TimestampedModel):

    objects = ProblemQuerySet.as_manager()

    NOBODY_ON_DUTY = 'nobody_on_duty'
    STAFF_HAS_GAP = 'staff_has_gap'
    NEW_MEMBER_IN_SCHEDULE = 'new_member_in_schedule'
    REASONS = (
        (NOBODY_ON_DUTY, _('Дежурный не назначен')),
        (STAFF_HAS_GAP, _('У дежурного есть незакрытые заменами отсутствия')),
        (NEW_MEMBER_IN_SCHEDULE, _('В дежурстве появился новый участник')),
    )
    HUMAN_TEXT = {
        NOBODY_ON_DUTY: {
            'ru': 'Дежурный не назначен',
            'en': 'Nobody is duty on the shift',
        },
        STAFF_HAS_GAP: {
            'ru': 'Дежурный({} {}) не может дежурить всю смену',
            'en': 'The person on duty({} {}) cannot be on duty all the shift',
        },
        NEW_MEMBER_IN_SCHEDULE: {
            'ru': 'В дежурстве появился новый участник',
            'en': 'There is new member in schedule'
        },
    }

    NEW = 'new'
    REPORTED = 'reported'
    RESOLVED = 'resolved'
    STATUSES = (
        (NEW, _('Новая проблема')),
        (REPORTED, _('Пользователи оповещены о проблеме')),
        (RESOLVED, _('Проблема решена')),
    )

    reason = models.TextField(
        choices=REASONS,
        verbose_name=_('код проблемы'),
        db_index=True,
    )
    shift = models.ForeignKey(
        Shift,
        null=True,
        blank=True,
        verbose_name=_('Смена, в которой проблема'),
        related_name='problems',
        on_delete=models.CASCADE,
        db_index=True,
    )
    schedule = models.ForeignKey(
        Schedule,
        null=True,
        blank=True,
        verbose_name=_('График, в котором проблема'),
        related_name='problems',
        on_delete=models.CASCADE,
        db_index=True,
    )
    staff = models.ForeignKey(
        Staff,
        null=True,
        blank=True,
        verbose_name=_('Новый участник в графике'),
        on_delete=models.CASCADE,
    )
    status = models.TextField(
        choices=STATUSES,
        default=NEW,
        verbose_name=_('Статус проблемы'),
        db_index=True,
    )
    datetime = models.DateTimeField(null=True, blank=True, db_index=True, verbose_name=_('Время проблемы'))
    report_date = models.DateTimeField(null=True, blank=True, verbose_name=_('Дата оповещения о проблеме'))
    resolve_date = models.DateTimeField(null=True,  blank=True, verbose_name=_('Дата закрытия проблемы'))

    @classmethod
    def open_shift_problem(cls, shift, reason, datetime):
        shift_problems = cls.objects.filter(shift=shift).active(reason)
        if not shift_problems.exists():
            return cls.objects.create(shift=shift, reason=reason, datetime=datetime)
        else:
            existing_problem = shift_problems.first()
            if existing_problem.datetime is None or existing_problem.datetime > datetime:
                existing_problem.datetime = datetime
                existing_problem.save(update_fields=['datetime'])
            return existing_problem

    @classmethod
    def open_schedule_new_member(cls, schedule, staff_id):
        reason = Problem.NEW_MEMBER_IN_SCHEDULE
        if not cls.objects.filter(schedule=schedule, staff_id=staff_id, status=cls.NEW, reason=reason).exists():
            return Problem.objects.create(schedule=schedule, reason=reason, staff_id=staff_id)

    @property
    def human_text(self):
        if self.reason == self.STAFF_HAS_GAP:
            text = self.HUMAN_TEXT[self.reason]
            staff = self.shift.staff
            if staff is not None:
                return {
                    'ru': text['ru'].format(staff.first_name, staff.last_name),
                    'en': text['en'].format(staff.first_name_en, staff.last_name_en),
                }
            else:
                return self.HUMAN_TEXT[self.NOBODY_ON_DUTY]
        else:
            return self.HUMAN_TEXT[self.reason]


class DutyToWatcher(models.Model):
    abc_id = models.IntegerField()
    watcher_id = models.IntegerField()
