from collections import defaultdict
from itertools import chain
import logging

from django.conf import settings
from django.db import transaction

from plan.common.utils import timezone as utils
from plan.duty.models import Gap, Shift, Schedule
from plan.duty.schedulers.ordered_staff import OrderedStaff
from plan.duty.schedulers.scheduler import DutyScheduler, DutySchedulerRegister
from plan.duty.utils import gap_localize
from plan.holidays.utils import end_datetime_to_end_date
from plan.staff.models import Staff

logger = logging.getLogger(__name__)


@DutySchedulerRegister.register
class AutoOrderingScheduler(DutyScheduler):

    priority = 0
    unique_for_schedule = False

    @classmethod
    def compatible_with_schedule(cls, schedule):
        return schedule.is_no_order_consider()

    def __init__(self, service, schedule=None):
        self.service = service
        if schedule:
            self.schedules = [schedule]
        else:
            self.schedules = list(self.service.schedules.active().with_recalculation().select_related('role').no_order_consider())
        self.schedules_to_staff = {}
        for schedule in self.schedules:
            members = self.service.members.of_schedule(schedule)
            self.schedules_to_staff[schedule] = list(Staff.objects.filter(pk__in=members.values_list('staff')))

        def get_schedule_key(data):
            coefficient = len(self.schedules_to_staff[data]) / float(data.persons_count)
            return -data.duration, coefficient, data.slug

        self.schedules.sort(key=get_schedule_key)
        self.gaps_by_users = defaultdict(list)
        self.fill_gaps()
        self.shifts_by_users = defaultdict(dict)
        self.fill_shifts()

    def fill_gaps(self):
        now = utils.today()
        max_date = now + settings.DUTY_SCHEDULING_PERIOD
        staffs = chain(*self.schedules_to_staff.values())
        gaps = gap_localize(
            Gap.objects
            .active()
            .filter(staff__in=staffs, start__date__lte=max_date, end__date__gte=now, work_in_absence=False)
            .order_by('start')
            .values('staff_id', 'start', 'end', 'full_day')
        )
        for gap in gaps:
            self.gaps_by_users[gap['staff_id']].append(gap)

    def fill_shifts(self):
        now = utils.today()
        max_date = now + settings.DUTY_SCHEDULING_PERIOD
        shifts = Shift.objects.filter(start__lte=max_date, end__gte=now, schedule__in=self.schedules)
        for shift in shifts:
            self.shifts_by_users[shift.staff_id][shift.id] = shift

    def update_shifts_by_user(self, staff, shift):
        self.shifts_by_users[getattr(staff, 'id', None)][shift.id] = shift

    def get_gaps(self, staff_id, start, end):
        return [gap for gap in self.gaps_by_users[staff_id] if gap['start'] <= end and gap['end'] >= start]

    def _load_ordered_staff(self, schedule):
        all_staff = self.schedules_to_staff[schedule]
        ordered_staff = OrderedStaff()
        past_shifts = (
            schedule.shifts.past_shifts()
            .fulltime()
            .filter(staff__in=all_staff)
            .order_by('-start')
            .select_related('staff')
        )
        staff_to_date = {}
        for shift in past_shifts:
            if len(all_staff) == len(staff_to_date):
                break
            if shift.staff and shift.staff not in staff_to_date:
                staff_to_date[shift.staff] = shift.end_datetime
        ordered_staff.bulk_update(staff_to_date)
        missed_staff = set(all_staff) - set(ordered_staff.all_humans())
        ordered_staff.add_staff_with_fake_date(missed_staff)
        return ordered_staff

    def _get_next_staff(self, ordered_staff, exclude_staff, shift=None, dates=None, schedule=None):
        assert shift or dates
        if shift:
            start = shift.start_datetime
            end = shift.end_datetime
            shift.schedule
        else:
            start, end = dates
        for staff in ordered_staff:
            if staff.id in exclude_staff:
                continue
            have_shifts_in_same_time = self.have_shifts_in_same_time(staff, shift, start, end)
            user_is_available, _ = self.is_available(staff.id, start, end, shift)
            if user_is_available and not have_shifts_in_same_time:
                return staff

    def have_shifts_in_same_time(self, staff, shift, start, end):
        shifts = {
            shift
            for shift
            in self.shifts_by_users[staff.id].values()
            if (shift.end_datetime > start >= shift.start_datetime)
            or (shift.end_datetime >= end > shift.start_datetime)
            or (start <= shift.start_datetime < end)
            or (start < shift.end_datetime <= end)
        }
        shifts.discard(shift)
        return bool(shifts)

    def create_one_shift(self, schedule, start_datetime, end_datetime, ordered_staff, exclude_staff, index):
        staff = self._get_next_staff(
            ordered_staff,
            exclude_staff,
            dates=(start_datetime, end_datetime),
            schedule=schedule,
        )
        now = utils.now()
        if start_datetime < now:
            logger.error(
                'Create shift for schedule %s with start %s less than current time %s',
                schedule.id, start_datetime, now,
            )
        if staff:
            ordered_staff.add_staff_with_date(staff, end_datetime)
        shift = Shift.objects.create(
            staff=staff,
            start=start_datetime.astimezone(settings.DEFAULT_TIMEZONE).date(),
            start_datetime=start_datetime,
            end=end_datetime_to_end_date(end_datetime.astimezone(settings.DEFAULT_TIMEZONE)),
            end_datetime=end_datetime,
            schedule=schedule,
        )
        self.update_shifts_by_user(staff, shift)
        return shift

    def update_shift_staff_if_needed(self, shift, exclude_staff, ordered_staff, shift_is_current=False):
        shift_updated = False
        # текущие шифты не трогаем
        if not shift_is_current:
            next_staff = self._get_next_staff(ordered_staff, exclude_staff, shift=shift)
            if shift.staff is None and shift.is_approved:
                shift.set_approved(False, None, save=False)
                shift_updated = True

            if shift.staff is None or not shift.is_approved and shift.staff != next_staff:
                shift.staff = next_staff
                shift_updated = True

        if shift.replace_for is None:
            if ordered_staff._staff_to_date.get(shift.staff):
                ordered_staff.add_staff_with_date(shift.staff, shift.end_datetime)
                self.update_shifts_by_user(shift.staff, shift)
        return shift_updated

    def process_created_shifts(self, removed_users=None):
        if removed_users:
            shifts = Shift.objects.future_and_present().filter(schedule__in=self.schedules, staff__in=removed_users)
            shifts.fulltime().update(staff=None, is_approved=False, approved_by=None, approve_datetime=None)
            shifts.parttime().delete()
            self.fill_shifts()

    def exclude_staff(self, schedule, start_datetime):
        if schedule.allow_sequential_shifts:
            exclude_staff = set()
        else:
            exclude_staff = self.get_staff_of_previous_iteration(schedule, start_datetime)

        return exclude_staff

    @transaction.atomic
    def create_new_shifts(self):
        shifts_to_update = (
            Shift.objects.filter(is_approved=False, schedule__in=self.schedules)
            .not_started()
            .future()
            .fulltime()
        )
        shifts_to_update.update(staff=None)
        self.fill_shifts()
        for schedule in self.schedules:
            """
                Берем всех возможных дежурных и делаем сопоставление {человек: конец-последней-сманы} в ordered_staff
                При этом учитываем только смены в прошлом
            """
            ordered_staff = self._load_ordered_staff(schedule)

            current_shifts = list(schedule.shifts.current_shifts())
            for shift in current_shifts:
                self.update_shift_staff_if_needed(shift, exclude_staff=None, ordered_staff=ordered_staff, shift_is_current=True)
                self.update_shift_problems(shift, save_if_updated=True)

            self.process_schedule(schedule, ordered_staff)

    def _replace_shift_staff(self, shift, exclude_staff, schedule):
        staff_not_used = {staff.id for staff in self.schedules_to_staff[schedule]} - exclude_staff
        previous_shifts = (
            schedule.shifts.filter(start_datetime__lt=shift.start_datetime)
            .filter(staff__in=self.schedules_to_staff[schedule])
            .order_by('-start')
            .select_related('staff')
        )
        ordered_staff = OrderedStaff()
        for shift in previous_shifts:
            if shift.staff in staff_not_used:
                staff_not_used.remove(shift.staff)
                ordered_staff.add_staff_with_date(shift.staff, shift.end_datetime)
                if not staff_not_used:
                    break
        if staff_not_used:
            ordered_staff.add_staff_with_fake_date(staff_not_used)
        self.shifts_by_users[shift.staff].pop(shift.id, None)
        shift.staff = self._get_next_staff(ordered_staff, exclude_staff, shift=shift)
        self.shifts_by_users[shift.staff][shift] = shift

    def get_staff_of_previous_iteration(self, schedule, date):
        start = (
            schedule.shifts
            .fulltime()
            .filter(start_datetime__lt=date)
            .order_by('-start')
            .values_list('start_datetime', flat=True)
            .first()
        )
        return set(schedule.shifts.filter(start_datetime=start).values_list('staff_id', flat=True))

    def check_one_shift(self, shift, schedule):
        """
        Проверяем может ли дежурить человек и заменяем его, если необходимо
        """
        update_fields = set()
        if self.update_shift_problems(shift, False):
            update_fields.add('has_problems')

        condition = (
            not shift.is_approved
            and not self.shift_will_start_soon(shift)
        )
        exclude_staff = {shift.staff_id}
        if not schedule.allow_sequential_shifts:
            previous_staff = self.get_staff_of_previous_iteration(shift.schedule, shift.start_datetime)
            exclude_staff = exclude_staff | previous_staff

        if condition:
            self._replace_shift_staff(shift, exclude_staff, schedule)
            update_fields.add('staff')

        if 'staff' in update_fields and self.update_shift_problems(shift, False):
            update_fields.add('has_problems')
        if update_fields:
            shift.save(update_fields=update_fields)


@DutySchedulerRegister.register
class AutoOrderingUniqueScheduler(AutoOrderingScheduler):

    priority = 1
    unique_for_schedule = True

    @classmethod
    def compatible_with_schedule(cls, schedule):
        return schedule.algorithm == Schedule.NO_ORDER and not schedule.consider_other_schedules
