import logging
from django.conf import settings

from plan.common.utils import timezone as utils
from plan.duty.models import Schedule, Shift
from plan.duty.schedulers.scheduler import DutyScheduler, DutySchedulerRegister
from plan.holidays.utils import end_datetime_to_end_date

logger = logging.getLogger(__name__)


@DutySchedulerRegister.register
class ManualOrderingScheduler(DutyScheduler):

    priority = 1
    unique_for_schedule = True

    @classmethod
    def compatible_with_schedule(cls, schedule):
        return schedule.algorithm == Schedule.MANUAL_ORDER

    def __init__(self, service, schedule):
        self._staffs = [
            order.staff
            for order in schedule.orders.order_by('order').select_related('staff')
        ]
        self._actual_shifts = schedule.shifts.fulltime().order_by('start')
        self._shift_duration = schedule.duration
        self._default_start_date = utils.make_localized_datetime(schedule.start_date, schedule.start_time)
        self._schedule = schedule
        self._persons_count = schedule.persons_count
        self._offset = schedule.manual_ordering_offset

    def update_shift_staff_if_needed(self, shift, exclude_staff=None, ordered_staff=None, remove_user=False, shift_is_current=False):
        """ Автоматическое обновление дежурного для шифта
            Не обновляем дежурного, если шифт уже начался или подтвержден руководителем
            (кроме обнуления дежурного если он был удален из графика)
        """
        if not remove_user and (shift.state == Shift.STARTED or shift.is_approved):
            return False
        staff = None
        if self._staffs and not remove_user:
            staff = self._staffs[(shift.index + self._offset) % len(self._staffs)]
        changed = shift.staff != staff
        shift.staff = staff
        return changed

    def get_start_end_for_shift(self, shift, previous_shift=None):
        if shift.index % self._persons_count == 0:
            return
        else:
            return previous_shift.start_datetime, previous_shift.end_datetime

    def validating_duplicate_shifts(self, schedule, already_created_shifts, index):
        """
        Если дублируются, проверим на соответствие порядку.
        """
        # пофильтровать по индексу и сравнить с пришешдшим
        deleted_duplicate = []
        not_duplicate =[]
        max_index = index + schedule.persons_count
        for shift in sorted(already_created_shifts, key=lambda s: s.index):
            if shift.index == index and index < max_index:
                not_duplicate.append(shift)
                index += 1

            else:
                deleted_duplicate.append(shift)

        logger.info(
            f'Deleted duplicate shifts {deleted_duplicate} for schedule {schedule} with id={schedule.id}'
        )
        schedule.shifts.filter(pk__in=[shift.pk for shift in deleted_duplicate]).delete()
        return not_duplicate

    def create_one_shift(self, schedule, start_datetime, end_datetime, ordered_staff, exclude_staff, index):
        shift = Shift(
            index=index,
            start=start_datetime.astimezone(settings.DEFAULT_TIMEZONE).date(),
            start_datetime=start_datetime,
            end=end_datetime_to_end_date(end_datetime.astimezone(settings.DEFAULT_TIMEZONE)),
            end_datetime=end_datetime,
            schedule=schedule,
        )
        self.update_shift_staff_if_needed(shift)
        shift.save()
        return shift

    def create_new_shifts(self):
        if self._schedule.recalculate:
            self.process_schedule(self._schedule)

    def process_created_shifts(self, removed_users=None):
        shifts = self._actual_shifts.filter(end_datetime__gt=utils.now())
        remove_user = False

        if removed_users is not None:
            shifts = shifts.filter(staff__in=removed_users)
            shifts.update(is_approved=False, approved_by=None, approve_datetime=None)
            remove_user = True

        for shift in shifts:
            updated = self.update_shift_staff_if_needed(shift, remove_user=remove_user)
            updated |= self.update_shift_problems(shift, save_if_updated=False)
            if updated:
                shift.save(update_fields=['staff', 'has_problems'])

    @classmethod
    def get_offset(cls, schedule, staff):
        scheduler = cls(schedule.service, schedule)

        shift_index = 0
        active_shift = schedule.shifts.current_shifts().fulltime().order_by('index').last()
        if active_shift:
            # Берем следующий шифт после активного с наибольшим индексом
            if active_shift.index is None:
                shifts = schedule.shifts.fulltime().order_by('start')
                for num, shift in enumerate(shifts):
                    shift.index = num
                    shift.save(update_fields=['index'])

                active_shift = schedule.shifts.current_shifts().fulltime().order_by('index').last()
            shift_index = active_shift.index + 1

        staff_index_without_offset = shift_index % len(scheduler._staffs)
        staff_index_must_be = scheduler._staffs.index(staff)
        return (staff_index_must_be - staff_index_without_offset) % len(scheduler._staffs)

    @classmethod
    def prepare_one(cls, service, schedule, use_full_recalculation, removed_users):
        role_q = schedule.get_role_q()
        correct_staffs = service.members.filter(role_q).values_list('staff', flat=True)
        schedule.orders.exclude(staff__in=correct_staffs).update(staff=None)
