from collections import deque

from django.conf import settings
from django.utils import timezone


class OrderedStaff(object):
    """ Класс используется в AutoOrderingScheduler для определения следующей кандидатуры для смены
    Интерфейс:
        add_staff_with_date(staff: Staff, date: Date) - добавить человека / обновить дату последнего дежурства
        bulk_update(staff_dates: list(tuple(Staff, Date))) - балковый вариант предыдущей функции
        add_staff_with_fake_date(staff_list: list(Staff)) - добавить несколько человек в класс и максимально ранюю дату
        all_humans(): list(Staff) - все используемые люди
        count(): int - кол-во используемых людей
        __iter__() - итерирование по людям в порядке приоритета
    """

    def __init__(self):
        self._staff_to_date = {}
        self._queue = deque()

    def add_staff_with_date(self, staff, date):
        if staff is None:
            return
        previous_date = self._staff_to_date.get(staff)
        if not previous_date or not previous_date > date:
            self._staff_to_date[staff] = date
            if len(self._queue) and staff == self._queue[0]:
                self._step_queue()
            else:
                self._rebuild_queue()

    def bulk_update(self, staff_dates):
        self._staff_to_date.update(staff_dates)
        self._rebuild_queue()

    def add_staff_with_fake_date(self, staff_list):
        fake_date = timezone.datetime(1970, 1, 1, 0, 0, tzinfo=settings.DEFAULT_TIMEZONE)
        for staff in staff_list:
            self._staff_to_date[staff] = fake_date
        self._rebuild_queue()

    def all_humans(self):
        return self._staff_to_date.keys()

    def count(self):
        return len(self._staff_to_date)

    def __iter__(self):
        return self._queue.__iter__()

    def _step_queue(self):
        staff = self._queue.popleft()
        self._queue.append(staff)

    def _rebuild_queue(self):
        value = [(time, staff) for staff, time in self._staff_to_date.items()]
        value.sort(key=lambda x: x[0])
        self._queue = deque(staff for _, staff in value)
