import abc
import logging

from collections import defaultdict

import datetime
from django.conf import settings
from django.utils import timezone
from django.db.models import Exists, OuterRef, Q

from plan.common.utils import timezone as utils
from plan.common.utils.timezone import make_localized_datetime
from plan.duty.models import Shift, Schedule, Gap, Problem
from plan.duty.utils import gap_localize
from plan.holidays.models import Holiday
from plan.holidays.utils import workday, next_workday, trim_holidays_and_weekends

logger = logging.getLogger(__name__)


class DutySchedulerRegister(object):

    schedulers_by_priority = []

    @classmethod
    def register(cls, scheduler_class):
        cls.schedulers_by_priority.append(scheduler_class)
        cls.schedulers_by_priority.sort(key=lambda scheduler: scheduler.priority, reverse=True)
        return scheduler_class


class DutyScheduler(object):

    @property
    @abc.abstractmethod
    def unique_for_schedule(self):
        pass

    @property
    @abc.abstractmethod
    def priority(self):
        pass

    @abc.abstractmethod
    def process_created_shifts(self):
        pass

    @abc.abstractmethod
    def create_new_shifts(self):
        pass

    @classmethod
    def compatible_with_schedule(cls, schedule):
        raise NotImplementedError()

    @classmethod
    def initialize_scheduler_by_schedule(cls, schedule):
        for scheduler_class in DutySchedulerRegister.schedulers_by_priority:
            if scheduler_class.compatible_with_schedule(schedule):
                service = schedule.service
                if scheduler_class.unique_for_schedule:
                    return scheduler_class(service, schedule=schedule)
                else:
                    return scheduler_class(service)

    @classmethod
    def initialize_scheduler_by_shift(cls, shift):
        return cls.initialize_scheduler_by_schedule(shift.schedule)

    @classmethod
    def find_active_segments(cls, start, end, shift):
        start_dt = start

        # Из-за использования времени начала дежурства, реально дежурство закончится не в день end,
        # а на следующий день в start_time часов
        end_dt = end
        if shift is None:
            return [{'start': start_dt, 'end': end_dt}]

        inactive_intervals = list(
            Shift.objects
            .filter(replace_for=shift)
            .order_by('start')
            .values_list('start_datetime', 'end_datetime')
        )

        # Для графиков без учета праздников и/или выходных, их нужно исключить, поэтому добавляем их в список интервалов
        # Потом сортируем интервалы в порядке возрастания начал, а при их равенстве в порядке убывания длин
        inactive_intervals += [
            (
                make_localized_datetime(holiday.date, datetime.time()),
                make_localized_datetime(holiday.date, datetime.time()) + timezone.timedelta(days=1)
            )
            for holiday in Holiday.objects.filter(date__gte=start.date(), date__lte=end.date())
            if not holiday.is_workday(shift.schedule.duty_on_holidays, shift.schedule.duty_on_weekends)
        ]
        inactive_intervals.sort(key=lambda i: (i[0], i[1] - i[0]))

        if len(inactive_intervals) == 0:
            return [{'start': start_dt, 'end': end_dt}]
        segments_shift = []
        for interval_start, interval_end in inactive_intervals:
            if start_dt < interval_start:
                segments_shift.append({'start': start_dt, 'end': interval_start})
            start_dt = interval_end

        # если есть, что положить ещё
        if start_dt <= end_dt:
            segments_shift.append({'start': start_dt, 'end': end_dt})

        return segments_shift

    def absence_timedelta(self, segment, staff_id, start, end):
        """
        Считаем отсуствия за сегмент
        """

        not_available_timedelta = timezone.timedelta()
        first_absence = None

        segment_start_dt = segment['start']
        segment_end_dt = segment['end']

        now = timezone.now()

        for gap in self.get_gaps(staff_id, start, end+timezone.timedelta(days=1)):
            if gap['end'] < segment_start_dt:
                continue
            elif gap['start'] >= segment_end_dt:
                break

            if segment_start_dt <= gap['end']:
                start_dt = max(segment_start_dt, gap['start'])
                end_dt = min(segment_end_dt, gap['end'])
                if end_dt >= now:
                    not_available_timedelta += end_dt - start_dt
                    first_absence = min(first_absence, start_dt) if first_absence else start_dt

        return not_available_timedelta, first_absence if not_available_timedelta != timezone.timedelta() else None

    @classmethod
    def update_shift_staff(cls, shift, staff, save=True, send_email=False):
        was_started = (shift.state == Shift.STARTED)
        if was_started:
            shift.cancel()

        shift.staff = staff

        if was_started:
            shift.begin()

        if save:
            shift.save(update_fields=['state', 'staff'])

    def update_shift_problems(self, shift, save_if_updated=True, check_replaces=True):
        has_problems = False
        if check_replaces:
            for replace in shift.replaces.all():
                self.update_shift_problems(replace, save_if_updated=True, check_replaces=False)
                has_problems |= replace.has_problems
        if shift.staff_id is None:
            shift.problems.active(Problem.STAFF_HAS_GAP).set_resolved()
            Problem.open_shift_problem(shift, Problem.NOBODY_ON_DUTY, shift.start_datetime)
            has_problems = True
        else:
            staff_available, first_absence = self.is_available(
                shift.staff_id, shift.start_datetime, shift.end_datetime, shift
            )
            if not staff_available:
                shift.problems.active(Problem.NOBODY_ON_DUTY).set_resolved()
                Problem.open_shift_problem(shift, Problem.STAFF_HAS_GAP, first_absence)
                has_problems = True
            else:
                shift.problems.active().set_resolved()

        updated = shift.has_problems != has_problems
        shift.has_problems = has_problems

        if updated and save_if_updated:
            shift.save(update_fields=['has_problems'])

        return updated

    @staticmethod
    def shift_will_start_soon(shift):
        duration = shift.schedule.duration
        return (shift.staff is not None) and (shift.start_datetime < utils.now() + 2 * duration)

    def is_available(self, staff_id, start, end, shift=None):
        """
        Проверяем есть ли отсутствия у staff в период [start, end]
        """
        # находим отрезки без замен
        segments_shift = self.find_active_segments(start, end, shift)

        not_available_timedelta = timezone.timedelta()
        first_absence = None
        for segment in segments_shift:
            segment_timedelta, segment_first_absence = self.absence_timedelta(segment, staff_id, start, end)
            not_available_timedelta += segment_timedelta

            if segment_first_absence:
                first_absence = min(first_absence, segment_first_absence) if first_absence else segment_first_absence

        return not_available_timedelta < timezone.timedelta(minutes=10), first_absence

    def get_gaps(self, staff_id, start, end):
        return gap_localize(list(
            Gap.objects
            .active()
            .filter(
                staff_id=staff_id,
                start__date__lte=end,
                end__date__gte=start,
                work_in_absence=False,
            )
            .order_by('start')
            .values('start', 'end', 'full_day')
        ))

    @staticmethod
    def check_weekends(schedule, start_datetime, end_datetime):
        """
        Сдвигаем start и end, если между ними есть выходные и праздники в зависимости от настроек графика
        duty_on_holidays и duty_on_weekends
        Если у нас дефолтные настройки начала смен - 00:00 по Мск,
        то не перекрываем выходные.
        """

        if not schedule.duty_on_holidays or not schedule.duty_on_weekends:
            start_datetime, delta = next_workday(start_datetime, schedule.duty_on_holidays, schedule.duty_on_weekends)
            end_datetime += delta

            qs_holiday = Holiday.objects.filter(date__range=(start_datetime, end_datetime))
            if not schedule.duty_on_holidays and not schedule.duty_on_weekends:
                # если оба флага выставлены False, то нерабочими считаются любые объекты Holiday
                num_of_holidays_covered = qs_holiday.count()

            else:
                # если один из флагов всё-таки False, то остаётся два варианта:
                # * если дежурим по выходным, то нерабочие - праздники
                # * если дежурим по праздникам, то нерабочие - выходные
                num_of_holidays_covered = qs_holiday.filter(is_holiday=schedule.duty_on_weekends).count()

            num_of_workdays_covered = (end_datetime - start_datetime).days - num_of_holidays_covered

            while num_of_workdays_covered < schedule.duration.days:
                end_datetime += timezone.timedelta(days=1)
                if workday(end_datetime, schedule.duty_on_holidays, schedule.duty_on_weekends):
                    num_of_workdays_covered += 1

            end_datetime = trim_holidays_and_weekends(end_datetime, schedule.duty_on_holidays, schedule.duty_on_weekends)

        return start_datetime, end_datetime

    @classmethod
    def prepare_one(cls, service, schedule, use_full_recalculation, removed_users):
        pass

    @classmethod
    def update_problems_no_recalculate(cls, service):
        schedules = service.schedules.active().filter(recalculate=False).values_list('id', flat=True)
        gap = (
            Gap.objects.filter(staff=OuterRef('staff'))
            .filter(
                Q(start__gte=OuterRef('start'))
                | Q(end__lte=OuterRef('end'))
            )
        )

        shifts = (
            Shift.objects.future_and_present()
            .filter(schedule_id__in=schedules)
            .annotate(staff_has_gap=Exists(gap))
        )

        without_staff = Q(
            staff__isnull=True
        )

        staff_has_gap = Q(
            staff_has_gap=True
        )

        with_problem = Q(
            has_problems=True
        )

        possibly_problematic_shift = shifts.filter(without_staff | staff_has_gap | with_problem)
        for shift in possibly_problematic_shift:
            # update_shift_problems у обоих алгоритмов один и тот же
            DutyScheduler().update_shift_problems(shift=shift, save_if_updated=True)

    @classmethod
    def recalculate_shifts(cls, service, full_recalculate_schedules_ids=None, removed_users=None):
        full_recalculate_schedules_ids = full_recalculate_schedules_ids or []
        if len(full_recalculate_schedules_ids) > 0:
            shifts = Shift.objects.future().filter(
                schedule__in=full_recalculate_schedules_ids,
                replace_for=None,
            ).exclude(schedule__algorithm=Schedule.MANUAL_ORDER)
            if removed_users:
                shifts = shifts.filter(Q(is_approved=False) | Q(staff__in=removed_users))
            else:
                shifts = shifts.filter(is_approved=False)
            shifts.delete()

        schedules = list(service.schedules.active().with_recalculation())
        for scheduler_class in DutySchedulerRegister.schedulers_by_priority:
            matched, missmatched = [], []

            for schedule in schedules:
                if scheduler_class.compatible_with_schedule(schedule):
                    matched.append(schedule)
                else:
                    missmatched.append(schedule)
            schedules = missmatched
            if scheduler_class.unique_for_schedule:
                for schedule in matched:
                    use_full_recalculation = schedule.id in full_recalculate_schedules_ids
                    scheduler_class.prepare_one(
                        service,
                        schedule,
                        use_full_recalculation,
                        removed_users,
                    )
                    scheduler = scheduler_class(service, schedule)
                    scheduler.process_created_shifts(removed_users)
                    scheduler.create_new_shifts()
            else:
                scheduler = scheduler_class(service)
                scheduler.process_created_shifts(removed_users)
                scheduler.create_new_shifts()
        if schedules:
            logger.error("schedules in service %s has no scheduler: %s", service.id, schedules)

        # для графиков, которые не пересчитываемы, но активны, нужно обновить проблемы
        # нас интересуют либо смены:
        #   * где уже проблемы,
        #   * либо где нет стаффа
        #   * либо у стаффа есть гэп на это время
        cls.update_problems_no_recalculate(service)

    @staticmethod
    def get_last_end_for_schedule(schedule):
        last_end = make_localized_datetime(schedule.start_date, schedule.start_time)
        # индекс используется для ручного порядка
        index = 0

        shifts = schedule.shifts.current_shifts().fulltime().order_by('index')
        if not shifts:
            shifts = schedule.shifts.past_shifts().fulltime().order_by('index')

        for shift in shifts:
            if shift.end_datetime >= last_end:
                last_end = shift.end_datetime
            index = shift.index + 1 if shift.index is not None else index
        return last_end.astimezone(settings.DEFAULT_TIMEZONE), index

    def get_next_start_end(self, last_end, schedule):
        start_datetime = make_localized_datetime(last_end.astimezone(settings.DEFAULT_TIMEZONE), schedule.start_time)
        end_datetime = start_datetime + schedule.duration
        return self.check_weekends(schedule, start_datetime, end_datetime)

    def validating_duplicate_shifts(self, schedule, already_created_shifts, index):
        # не понятно: какие шифты из дублей считать важнее, затрём и создадим заново
        logger.info(
            f'Deleted duplicate shifts {already_created_shifts} for schedule {schedule} with id={schedule.id}'
        )
        schedule.shifts.filter(pk__in=[shift.pk for shift in already_created_shifts]).delete()
        return []

    def validating_existing_shifts(self, schedule, start_to_shift, start_datetime, index):
        """
        Созданные смены сохранены в start_to_shift.
        Если находим дату, нужную нам, то:
            * проверяем, что начало смен соответствует указанному времени в настройках,
                при необходимости помечаем флагом updated_start_end о необходимости пересохранить начало и конец смен
            * если минимальная найденная дата среди созданных смен несоотвествует ожиданиям, удаляем такие шифты
            * если количество созданных шифтов больше ожидаемого - удаляем дубликаты
        В конце удаляем обработанную дату из start_to_shift, чтобы там всегда лежала минимальная непроверенная.

        :return:
            already_created_shifts - список существующих шифтов для указанной даты
            updated_start_end - нужно ли апдейдить start_datetime/end_datetime
        """
        min_start = sorted(start_to_shift.keys())[0] if start_to_shift else None
        updated_start_end = False

        if min_start:
            if min_start != start_datetime and min_start.date() == start_datetime.date():
                # для пользователя это смена/смены, в которой/ых поменялось только время
                already_created_shifts = start_to_shift.pop(min_start)
                updated_start_end = True

            elif min_start < start_datetime:
                shifts = start_to_shift.pop(min_start)
                logger.info(
                    f'Deleted not relevant shifts {[s.id for s in shifts]} for schedule {schedule} with id={schedule.id}'
                )
                schedule.shifts.filter(pk__in=[shift.pk for shift in shifts]).delete()
                already_created_shifts = start_to_shift.pop(start_datetime, [])

            else:
                already_created_shifts = start_to_shift.pop(start_datetime, [])

        else:
            already_created_shifts = start_to_shift.pop(start_datetime, [])

        if len(already_created_shifts) > schedule.persons_count:
            # если есть дубликаты, надо это разрулить
            already_created_shifts = self.validating_duplicate_shifts(schedule, already_created_shifts, index)

        return already_created_shifts, updated_start_end, start_to_shift

    def exclude_staff(self, schedule, start_datetime):
        return set()

    @abc.abstractmethod
    def update_shift_staff_if_needed(self,
                                     shift,
                                     exclude_staff=None,
                                     ordered_staff=None,
                                     remove_user=False,
                                     shift_is_current=False
                                     ):
        """
        Возвращает True, если был изменен стафф в шифте.
        """

    @abc.abstractmethod
    def create_one_shift(self, schedule, start, end, ordered_staff, exclude_staff, index):
        """
        Создаёт и возвращает объект shift.
        """

    def process_schedule(self, schedule, ordered_staff=None):
        """
            Начинаем отсчет с активного дежурства (если есть) или с даты начала дежурства и на полгода вперёд:
                ordered_staff - актуально только для автопорядка
                Берем дату.
                    * Если есть уже созданные смены на эту дату без дежурного - назначаем дежурного, обноляем ordered_staff
                    * Если есть созданные смены с дежурным - только обновляем ordered_staff
                    * Создаём новые необходимые смены и обновляем ordered_staff

        """

        last_end, index = self.get_last_end_for_schedule(schedule)

        # Считаем смены
        # Собираем словарь datetime: [шифты]
        start_to_shift = defaultdict(list)
        for shift in schedule.shifts.future().fulltime().order_by('end', 'index'):
            start_to_shift[shift.start_datetime.astimezone(settings.DEFAULT_TIMEZONE)].append(shift)

        max_date = utils.now() + settings.DUTY_SCHEDULING_PERIOD
        start_datetime, end_datetime = self.get_next_start_end(last_end, schedule)

        while start_datetime < max_date:
            # посмотрим есть ли уже шифты на выбранную даты и проверим их состояние
            already_created_shifts, updated_start_end, start_to_shift = self.validating_existing_shifts(
                schedule, start_to_shift, start_datetime, index
            )

            # для ручного порядка обновим index
            index += len(already_created_shifts)

            exclude_staff = self.exclude_staff(schedule, start_datetime)

            for shift in already_created_shifts:
                if updated_start_end:
                    # флаг может быть установлен только при изменении времени начала, дата остаётся той же
                    shift.start_datetime = start_datetime
                    shift.end_datetime = end_datetime
                    shift.save(update_fields=['start_datetime', 'end_datetime'])

                # проверяем стафф на порядок и проблемы, обновляем если нужно
                updated_staff = self.update_shift_staff_if_needed(shift, exclude_staff, ordered_staff)
                updated_problem = self.update_shift_problems(shift, save_if_updated=False)

                if updated_staff or updated_problem:
                    shift.save(update_fields=[
                        'has_problems', 'staff', 'is_approved', 'approved_by', 'approve_datetime'
                    ])

            # если существующих шифтов не хватает, нужно создать
            for _ in range(schedule.persons_count - len(already_created_shifts)):
                shift = self.create_one_shift(
                    schedule, start_datetime, end_datetime, ordered_staff, exclude_staff, index
                )
                self.update_shift_problems(shift, save_if_updated=True, check_replaces=False)
                index += 1

            start_datetime, end_datetime = self.get_next_start_end(shift.end_datetime, schedule)
