import logging
import datetime
import time
import math

from typing import List, Tuple
from collections import defaultdict
from dateutil import parser

from django.conf import settings
from django.db import transaction
from django.db.models import F, Q, Max
from django.utils import timezone
from django.dispatch import receiver

from plan.api.idm import actions
from plan.celery_app import app
from plan.common.models import update_model_instance
from plan.common.utils import locks
from plan.common.utils import timezone as utils
from plan.common.utils.oauth import get_abc_zombik
from plan.common.utils.tasks import get_last_success_start, lock_task
from plan.duty.models import Gap, Schedule, Shift, Problem
from plan.duty.utils import get_gaps
from plan.duty.schedulers.scheduler import DutyScheduler
from plan.holidays.calendar import day_is_workdays
from plan.holidays.utils import not_workday
from plan.idm.exceptions import IDMError
from plan.roles.models import Role

from plan.services.models import (
    Service,
    ServiceMember,
    ServiceNotification,
)
from plan.services.tasks import send_notification_staffs, notify_staff_base
from plan.signals import reopen_service
from plan.staff.models import Staff

log = logging.getLogger(__name__)


RECALCULATE_SHIFTS_OR_SEND_PROBLEMS_LOCK_KEY = 'recalculate_shifts_or_send_problems_%s'
CHECK_CURRENT_SHIFTS_LOCK_KEY = 'check_current_shifts_%s'


@lock_task
def recalculate_all_duties():
    for service_id in Schedule.objects.active().values_list('service__pk', flat=True).distinct():
        default_recalculate_shifts_for_service.delay(service_id)


@transaction.atomic
def recalculate_duty_for_service(service_id, full_recalculate_schedules_ids=None, removed_users_ids=None, send_email=True,):
    service = Service.objects.prefetch_related('members').get(pk=service_id)
    if service.state not in Service.states.ACTIVE_STATES:
        return

    lock_name = RECALCULATE_SHIFTS_OR_SEND_PROBLEMS_LOCK_KEY % service_id
    log.info(
        'recalculate shifts planned with %s, %s, %s',
        lock_name,
        full_recalculate_schedules_ids,
        removed_users_ids,
    )
    with locks.locked_context(lock_name, block=True):
        log.info('recalculate shifts for %s started', service_id)
        DutyScheduler.recalculate_shifts(
            service,
            full_recalculate_schedules_ids,
            removed_users_ids,
        )
        autoapprove_shifts(service_id)
    notify_staff_duty.delay_on_commit(service.id, service.slug)


@app.task(bind=True)
def notify_staff_duty(self, service_id, service_slug):
    notify_staff_base(
        self,
        service_id,
        '{staff}/gap-api/api/need_update_gap/{service_id}/',
        json={
            'service_slug': service_slug,
        },
    )


@app.task
def priority_recalculate_shifts_for_service(
    service_id,
    full_recalculate_schedules_ids=None,
    removed_users_ids=None,
    send_email=True,
):
    """
    Эта таска запускается при создании или апдейте графика.
    Поэтому она должна попадать в более приоритетную очередь.
    """
    recalculate_duty_for_service(
        service_id,
        full_recalculate_schedules_ids,
        removed_users_ids,
        send_email,
    )


@app.task
def default_recalculate_shifts_for_service(
    service_id,
    full_recalculate_schedules_ids=None,
    removed_users_ids=None,
    send_email=False,
):
    """
    Эта таска запускается при регулярном пересчете.
    Нам не так принципиально, когда оно расчитается, поэтому поподает в дефолтную очередь.
    """
    recalculate_duty_for_service(
        service_id,
        full_recalculate_schedules_ids,
        removed_users_ids,
        send_email
    )


def split_shifts_to_sequences(shifts):
    """
    shifts с проблемами одного графика, отсортированые по старту.
    Сравниваем соседние шифты для склейки.
    Склеиваем и последовательные, и паралельные смены.
    """

    if not shifts:
        return []
    sequences = []
    previous_shift = shifts[0]
    current_sequence = [previous_shift]

    for shift in shifts[1:]:
        parallel_or_next = (
            shift.start_datetime == previous_shift.end_datetime or
            shift.start_datetime == previous_shift.start_datetime
        )
        if parallel_or_next and shift.has_equal_problems(previous_shift):
            current_sequence.append(shift)
        else:
            sequences.append(current_sequence)
            current_sequence = [shift]
        previous_shift = shift
    sequences.append(current_sequence)
    return sequences


def resolve_schedules_problems():
    Problem.objects.active().filter(
        reason=Problem.NEW_MEMBER_IN_SCHEDULE,
        schedule__orders__staff=F('staff')
    ).set_resolved()


def _problem_notifications(service, recipients, problem, **kwargs):
    notifications = []
    for recipient in recipients:
        notification = ServiceNotification(
            notification_id=ServiceNotification.DUTY_PROBLEM_NOTIFICATION,
            recipient=Staff.objects.get(id=recipient),
            service=service,
            problem=problem,
            **kwargs
        )
        notifications.append(notification)
    return notifications


def prepare_schedule_problems_notifications(service, recipients, schedule_problems):
    notifications = []
    for problem in schedule_problems:
        notifications.extend(_problem_notifications(service, recipients, problem))
    ServiceNotification.objects.bulk_create(notifications)
    Problem.objects.of_service(service).of_schedules().active().set_reported()


def notifications_for_shifts(service, recipients, shift, **kwargs):
    notifications = []
    days_before_shift = None
    if shift:
        days_before_shift = max((shift.start - utils.today()).days, 0)
    for problem in shift.problems.active():
        notifications.extend(
            _problem_notifications(
                service,
                recipients,
                problem,
                shift=shift,
                days_before_shift=days_before_shift,
                **kwargs
            )
        )
    return notifications


def prepare_shift_problems_notifications(service, recipients, shift_problems):
    problem_shifts_main = Shift.objects.alive().future_and_present().filter(problems__in=shift_problems, replace_for=None)
    problem_shifts_replacements = (
        Shift.objects.filter(problems__in=shift_problems, replace_for__isnull=False)
        .order_by('start')
    )
    sequences = []
    for schedule in problem_shifts_main.values_list('schedule', flat=True).distinct():
        sequences += split_shifts_to_sequences(
            list(problem_shifts_main
                 .filter(schedule=schedule)
                 .order_by('start'))
        )

    notifications = []

    for sequence in sequences:
        notifications.extend(
            notifications_for_shifts(
                service, recipients, sequence[0], last_shift=sequence[-1], spliced_shifts_count=len(sequence)
            )
        )
    for shift in problem_shifts_replacements:
        notifications.extend(notifications_for_shifts(service, recipients, shift))

    ServiceNotification.objects.bulk_create(notifications)
    shift_problems.set_reported()


def prepare_notifications_if_needed(service):
    log.info('preparing notifications for service: %s(%s)', service.slug, service.id)
    service_problems = Problem.objects.of_service(service).active()
    shift_problems = service_problems.of_shifts().future_and_present().only_nearest().active()
    shift_fresh_problems = shift_problems.ready_to_notification()
    # для проблем добавления новых людей в порядок должен быть статус новые
    new_member_problems = service_problems.of_schedules().with_enabled_notifications().new()
    today_is_holiday = not_workday(utils.today())
    if (
        not today_is_holiday and not shift_fresh_problems.exists() and not new_member_problems.exists() or
        today_is_holiday and not shift_fresh_problems.new().exists()
    ):
        return
    recipients = service.get_notifiable_about_duty().values_list('staff', flat=True).distinct()
    prepare_schedule_problems_notifications(service, recipients, new_member_problems)
    prepare_shift_problems_notifications(service, recipients, shift_problems)


@lock_task
def send_notifications_problems(service_id=None):
    # Пробуем разрешить проблемы нового человека
    resolve_schedules_problems()

    if service_id:
        services = Service.objects.filter(id=service_id)
    else:
        services = Service.objects.filter(id__in=(
            Schedule.objects
                    # это проблемы а не начало
                    .with_set_days_for_problem_notification()
                    .values_list('service_id', flat=True)
        )
        ).iterator()

    for service in services:
        lock_name = RECALCULATE_SHIFTS_OR_SEND_PROBLEMS_LOCK_KEY % service.id
        # Этот лок нужен для того, что б не запустить рассылку сервиса одновременно с пересчетом
        with locks.locked_context(lock_name, block=False) as acquired:
            if acquired:
                prepare_notifications_if_needed(service)
            else:
                log.info('preparing notification for service blocked %s(%s)', service.slug, service.id)
    send_notification_staffs(
        ServiceNotification.DUTY_PROBLEM_NOTIFICATION,
        'notifications.duty.shift_has_problems',
        order_by='shift__start',
    )


@app.task
def update_schedule(service_id, role_id, removed_user_id=None, added_user_id=None, remove_department=False):
    schedules_to_recalculate = (
        Schedule.objects
        .filter(service_id=service_id, status=Schedule.ACTIVE,)
        .with_recalculation()
        .for_role(role_id)
    )

    if removed_user_id:
        other_roles_of_removed_users = (
            ServiceMember.objects
            .filter(service_id=service_id)
            .filter(staff_id=removed_user_id)
            .exclude(Q(role__code__in=Role.CAN_NOT_USE_FOR_DUTY))
            .values_list('role', flat=True)
        )

        if other_roles_of_removed_users:
            schedules_to_recalculate = (
                schedules_to_recalculate
                .exclude(Q(role__in=other_roles_of_removed_users) | Q(role=None))
            )

        Shift.objects.parttime().filter(staff_id=removed_user_id, schedule__in=schedules_to_recalculate).delete()
        removed_user = Staff.objects.get(id=removed_user_id)
        if removed_user.is_dismissed:
            # текущие прямо проапдетим на нулевой стафф, тк они не поменяются при пересчете
            Shift.objects.current_shifts().filter(staff_id=removed_user_id).update(staff=None)
            # с будущих снимем подтверждение
            Shift.objects.future().filter(staff_id=removed_user_id).update(is_approved=False)
            # временные замены на этого человека удалим
            Shift.objects.parttime().filter(staff_id=removed_user_id).delete()

    full_schedules_id_set = set()
    for schedule in schedules_to_recalculate:
        full_schedules_id_set.update(schedule.get_associated_schedules_id_list(include_self=True))
        if added_user_id is None:
            continue

        schedule_role = schedule.get_role_q()
        user_in_orders = schedule.orders.filter(staff_id=added_user_id)
        membership_in_schedule = ServiceMember.objects.filter(
            schedule_role,
            staff_id=added_user_id,
            service=schedule.service,
        ).exclude(role_id=role_id)
        if (
            added_user_id
            and schedule.algorithm == Schedule.MANUAL_ORDER
            and not user_in_orders.exists()
            and not membership_in_schedule.exists()
        ):
            Problem.open_schedule_new_member(schedule, added_user_id)

    if remove_department:
        service = Service.objects.get(pk=service_id)
        for schedule_id in full_schedules_id_set:
            schedule = Schedule.objects.get(pk=schedule_id)
            correct_staff = service.members.of_schedule(schedule).values_list('staff', flat=True)
            shifts = schedule.shifts.with_staff().future().exclude(staff__in=correct_staff)
            shifts.parttime().delete()
            shifts.update(staff=None, is_approved=False)

    if schedules_to_recalculate:
        priority_recalculate_shifts_for_service.apply_async(args=[
            service_id,
            list(full_schedules_id_set),
            removed_user_id if removed_user_id is None else [removed_user_id],
        ])


@receiver(reopen_service, sender=Service)
def recalculate_for_reopen_service(sender, service, **kwargs):
    priority_recalculate_shifts_for_service.apply_async(
        args=[service.id],
        countdown=settings.ABC_DEFAULT_COUNTDOWN
    )


@lock_task(last_finished_metric_name='remove_inactive_schedules')
def remove_inactive_schedules():
    last_exception = None
    for schedule in Schedule.objects.deleted():
        try:
            shifts = schedule.shifts.started()
            for shift in shifts:
                shift.cancel()
                shift.delete()
            if timezone.now() > schedule.deleted_at + settings.SCHEDULE_DELETE_TIMEDELTA:
                schedule.delete()
        except Exception as exc:
            log.exception('failed on schedule %s', schedule.id)
            last_exception = exc
    if last_exception:
        raise last_exception


def set_shifts(today, notification_id):
    shifts = set()
    schedules = Schedule.objects.with_set_days_for_begin_notifications()
    for schedule in schedules:
        shifts |= set(
            schedule.shifts.for_notify_about_begin(today, notification_id)
            .filter(
                start__in=[
                    today + timezone.timedelta(days=day)
                    for day in schedule.days_for_begin_shift_notification
                ]
            )
        )
    return shifts


@lock_task
def send_notification_duty():
    today = utils.today()
    notification_id = ServiceNotification.DUTY_BEGIN_NOTIFICATION
    notifications = []

    schedules_days_for_begin = {
        schedule.id: schedule.days_for_begin_shift_notification
        for schedule in Schedule.objects.with_set_days_for_begin_notifications()
    }

    shifts = set_shifts(today, notification_id)

    # если сегодня рабочий и завтра выходной,
    # то найти смены, которые начинаются в следующий рабочий день
    tomorrow = today + timezone.timedelta(days=1)
    today_is_workday = day_is_workdays(today)
    if today_is_workday and not day_is_workdays(tomorrow):
        day = tomorrow
        while not day_is_workdays(day):
            add_shifts = set_shifts(day, notification_id)
            shifts |= add_shifts
            day += timezone.timedelta(days=1)

    for shift in shifts:
        # сегодняшная смена ещё не началась
        if shift.start == today and shift.start_datetime > utils.now():
            continue

        days_before, real_days_before_shift = shift.days_before_shift(sorted(schedules_days_for_begin[shift.schedule.id]))
        # если сегодня нерабочий день, то не отправляем ничего, кроме старта сегодня
        if not today_is_workday and days_before is not None and days_before > 0:
            continue

        notification = ServiceNotification(
            notification_id=notification_id,
            recipient=shift.staff,
            shift=shift,
            service=shift.schedule.service,
            days_before_shift=days_before,
            real_days_before_shift=real_days_before_shift
        )
        notifications.append(notification)

    ServiceNotification.objects.bulk_create(notifications)
    send_notification_staffs(
        notification_id,
        'notifications.duty.shift_start_or_soon',
        order_by='shift__start',
    )


def start_shifts(schedules):
    shifts = (
        Shift.objects
        .filter(schedule__in=schedules)
        .startable()
        .select_related('schedule', 'schedule__service', 'schedule__role_on_duty', 'staff', 'role')
    ).all()

    roles = {
        role.id: role
        for role in Role.objects.filter(id__in=shifts.values_list('cached_role_id'))
    }

    shifts_for_update = []
    for shift in shifts:
        # если смапили роль, то нужно просто проапдетить шифт на started
        if shift.member_id is not None:
            shifts_for_update.append(shift.id)
            shift.set_component_lead()
            continue

        try:
            role_data = actions.request_membership(
                shift.schedule.service,
                shift.staff,
                roles[shift.cached_role_id],
                comment='Начало дежурства',
                silent=True,
            )
        except IDMError as error:
            log.warning(f'Couldn\'t start shift {shift.id}: {error}', exc_info=True)
        else:
            member, _ = ServiceMember.all_states.get_or_create(
                service=shift.schedule.service,
                role=roles[shift.cached_role_id],
                staff=shift.staff,
                resource=None, from_department=None,
            )
            member.request(role_data['id'], autorequested=True)
            shift.set_component_lead()
            shifts_for_update.append(shift.id)

    if shifts_for_update:
        (
            shifts.filter(pk__in=shifts_for_update)
            .update(state=Shift.STARTED, role=F('cached_role_id'),)
        )


def finish_shifts(schedules):
    shifts = (
        Shift.objects
        .filter(schedule__in=schedules)
        .finishable()
        .select_related('staff', 'schedule', 'replace_for')
        .with_staff()
    )
    shifts_by_members = defaultdict(list)

    shifts_by_members_query = (
        shifts
        .annotate_intersecting().filter(intersecting=False)
        .annotate_member_id().filter(member_id__isnull=False)
    )

    for shift in shifts_by_members_query:
        shifts_by_members[shift.member_id].append(shift.id)
        if shift.replace_for and shift.replace_for.state == Shift.STARTED:
            shift.replace_for.set_component_lead()

    members = ServiceMember.objects.filter(
        id__in=shifts_by_members.keys(),
        autorequested=True,
    )

    with_error = []
    members.set_depriving_state()
    for member in members:
        try:
            actions.deprive_role(member, comment='Конец дежурства')
        except IDMError as error:
            log.warning(f'Couldn\'t finish shift(s) for member {member.id}: {error}', exc_info=True)
            with_error.extend(shifts_by_members[member.id])
    shifts.exclude(id__in=with_error).update(state=Shift.FINISHED)


@app.task
def check_current_shifts_for_service(service_id):
    lock_name = CHECK_CURRENT_SHIFTS_LOCK_KEY % service_id
    with locks.locked_context(lock_name, block=False):
        schedules = Schedule.objects.active().filter(service_id=service_id)
        start_shifts(schedules)
        finish_shifts(schedules)


@lock_task
def check_current_shifts():
    for service_id in Schedule.objects.active().values_list('service__pk', flat=True).distinct():
        check_current_shifts_for_service.delay(service_id)
    send_notification_duty.delay()


def get_logins_to_staff_pk_batches():
    start_pk = 0
    max_pk = Staff.objects.aggregate(Max('pk'))['pk__max']
    limit = 100
    while start_pk < max_pk:
        yield {staff.login: staff.pk for staff in Staff.objects.filter(pk__gt=start_pk, pk__lte=start_pk + limit)}
        start_pk += limit


@lock_task(last_finished_metric_name='sync_with_gap')
def sync_with_gap():
    today = utils.today()
    start = today - settings.GAP_PAST_SYNC_INTERVAL
    end = today + settings.DUTY_SCHEDULING_PERIOD
    last_exception = None
    for login_to_staff_pk in get_logins_to_staff_pk_batches():
        try:
            gaps_queryset = Gap.objects.filter(staff_id__in=login_to_staff_pk.values())
            saved_gaps = {gap.gap_id: gap for gap in gaps_queryset}
            real_gaps = get_gaps(list(login_to_staff_pk.keys()), start, end)
            gaps_to_create = []
            used_gaps_pks = []
            with transaction.atomic():
                for gap in real_gaps:
                    real_gap = Gap.from_gap_api_data(gap, staff_pk_cache=login_to_staff_pk)
                    saved_gap = saved_gaps.get(int(real_gap.gap_id))
                    if saved_gap is None:
                        gaps_to_create.append(real_gap)
                    else:
                        update_model_instance(
                            saved_gap,
                            real_gap,
                            {field.name for field in saved_gap._meta.fields} - {'created_at', 'id', 'staff'}
                        )
                        used_gaps_pks.append(saved_gap.pk)
                gaps_queryset.exclude(pk__in=used_gaps_pks).active().soft_delete()
                Gap.objects.bulk_create(gaps_to_create)
        except Exception as exc:
            log.exception('sync_with_gap raised exception')
            last_exception = exc

    if last_exception is not None:
        raise last_exception  # чтобы на графике было видно, что в таске что-то не так


@app.task
def offset_notify_staff_duty(services: List[Tuple[int, str]], offset: int = 0, limit: int = 100):
    # notify_staff_duty выполняется за 280ms
    # стандартный - 5с, берем 10
    countdown = settings.ABC_DEFAULT_COUNTDOWN * 2

    new_offset = offset + limit
    for service_id, service_slug in services[offset:new_offset]:
        notify_staff_duty.delay(service_id, service_slug)

    if new_offset < len(services):
        offset_notify_staff_duty.apply_async(
            args=[services, new_offset, limit],
            countdown=countdown
        )


@app.task
def autoapprove_shifts(service_id=None):
    if service_id is None:
        shifts = Shift.objects.all()
    else:
        shifts = Shift.objects.filter(schedule__service_id=service_id)

    autoapprove_time = timezone.now() + F('schedule__autoapprove_timedelta')

    ids_shifts_for_update = list(
        shifts.with_staff().scheduled().filter(
            is_approved=False,
            start_datetime__lte=autoapprove_time,
        ).values_list('id', flat=True)
    )

    Shift.objects.filter(pk__in=ids_shifts_for_update).update(
        is_approved=True,
        approved_by=get_abc_zombik(),
        approve_datetime=timezone.now(),
    )

    services = set(Service.objects.filter(schedules__shifts__in=ids_shifts_for_update))
    offset_notify_staff_duty.delay(list((service.id, service.slug) for service in services))


def get_dangling_duty_members():
    last_start = get_last_success_start('remove_dangling_duty_members')
    if last_start is None:
        last_start = utils.now() - timezone.timedelta(days=1)

    since_dt = last_start - timezone.timedelta(days=1)
    # тк роли отзываем через час (см настройку SHIFT_FINISH_AFTER_DUTY_START_TIMEDELTA) после окончания смены
    # то возьмём еще лаг в 1 час
    until_dt = utils.now() - settings.SHIFT_FINISH_AFTER_DUTY_START_TIMEDELTA - timezone.timedelta(hours=1)

    all_shift_member_ids, current_shift_member_ids = (
        shifts
        .with_staff()
        .annotate_member_id().exclude(member_id=None)
        .values_list('member_id', flat=True)
        .distinct()
        for shifts in [
            Shift.objects.filter(end_datetime__lt=until_dt, end_datetime__gte=since_dt).exclude(state=Shift.FINISHED),
            Shift.objects.current_shifts()
        ]
    )

    members_ids = set(all_shift_member_ids) - set(current_shift_member_ids)
    dangling_duty_members = (
        ServiceMember.objects
        .filter(
            autorequested=True,
            id__in=members_ids,
        )
    )
    return dangling_duty_members


@lock_task
def remove_dangling_duty_members():
    dangling_duty_members = get_dangling_duty_members().select_related('service', 'staff', 'role')
    log.info(f'Depriving {len(dangling_duty_members)} dangling duty roles')
    dangling_duty_members.update(state=ServiceMember.states.DEPRIVING)
    for member in dangling_duty_members:
        try:
            actions.deprive_role(member, comment='Неотозванная дежурная роль')
        except IDMError as error:
            log.warning(f"Couldn't deprive dangling duty role {member.id}: {error}")


@lock_task
def upload_duty_to_yt(for_date=None):
    for_date = parser.parse(for_date).date() if for_date else datetime.date.today() - datetime.timedelta(days=1)
    shifts = Shift.objects.filter(
        start__lte=for_date, end__gte=for_date,
        staff_id__isnull=False,
    ).exclude(state=Shift.CANCELLED).select_related(
        'staff', 'schedule__service',
        'schedule', 'replace_for',
    )
    items = set()
    for_date_min = settings.DEFAULT_TIMEZONE.localize(
        datetime.datetime.combine(for_date, datetime.datetime.min.time())
    )
    for_date_max = settings.DEFAULT_TIMEZONE.localize(
        datetime.datetime.combine(for_date, datetime.datetime.max.time())
    )
    for shift in shifts:
        if shift.replace_for and shift.replace_for.staff_id == shift.staff_id:
            # подсмена, но дежурный такой же как в основной
            # не будем учитывать, чтобы не задвоилось
            continue
        duty_from = max(for_date_min, shift.start_datetime)
        duty_to = min(for_date_max, shift.end_datetime)

        diff = (duty_to - duty_from)
        days, seconds = diff.days, diff.seconds
        hours = days * 24 + math.ceil(seconds / 3600)
        items.add(
            (
                shift.schedule.service.slug,
                shift.staff.login,
                shift.schedule.slug,
                hours,
            )
        )
    if items:
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)

        columns = (
            'service_slug',
            'staff_login',
            'schedule_slug',
            'duty_hours',
        )
        query = [
            'USE hahn;',
            'INSERT INTO',
            f'`home/abc/duty/{for_date.isoformat()}` with truncate',
            f'({",".join(columns)})',
            'VALUES'
        ]
        value_rows = []
        for item in items:
            value_row = []
            for value in item:
                if isinstance(value, int):
                    value_row.append(str(value))
                else:
                    value_row.append(f'"{value}"')
            value_rows.append(f'({", ".join(value_row)})')
        query.append(', '.join(value_rows))
        request = yql_client.query(' '.join(query))
        request.run()

        while request.status not in ('COMPLETED', 'ERROR'):
            log.debug(f'Export Duty data to Yql, request in status {request.status}')
            time.sleep(1.0)
        if request.status == 'ERROR':
            errors = '; '.join([error.format_issue() for error in request.errors])
            log.error(f'Error occurred on duty data export: [{errors}]')
            raise Exception('Upload duty data failed')


@lock_task
def upload_duty_future():
    """
    загружаем данные по дежурствам в YT на будущие даты
    нужно аналитикам для примерного рассчета
    """
    start = datetime.date.today()
    for delta in range(1, 180):
        upload_duty_to_yt(
            for_date=(start + datetime.timedelta(days=delta)).isoformat()
        )


@lock_task
def sync_important_schedules():
    from yql.api.v1.client import YqlClient

    yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)

    query = [
        'USE hahn;',
        'PRAGMA yt.InferSchema = \'1\';'
        'SELECT schedule, service',
        'FROM `home/test_assessor/zbp/service_duty_tier`',
    ]

    request = yql_client.query(' '.join(query))
    request.run()

    warden_map = defaultdict(set)

    for table in request.get_results():
        table.fetch_full_data()
        for row in table.rows:
            schedule_slug, service_slug = row
            warden_map[service_slug].add(schedule_slug)

    abc_map = defaultdict(set)
    schedule_map = {}
    current_important_schedules = Schedule.objects.select_related('service').filter(
        is_important=True
    )
    for schedule in current_important_schedules:
        abc_map[schedule.service.slug].add(schedule.slug)
        schedule_map[schedule.slug] = schedule.id

    to_add_important = []
    to_remove_important = []
    for service_slug, schedules in warden_map.items():
        to_add = schedules.difference(abc_map[service_slug])
        to_remove = abc_map[service_slug].difference(schedules)

        for schedule_slug in to_add:
            to_add_important.append((service_slug, schedule_slug))
        for schedule_slug in to_remove:
            to_remove_important.append(schedule_map[schedule_slug])

    for service_slug, schedule_slug in to_add_important:
        schedule = Schedule.objects.filter(
            slug=schedule_slug, service__slug=service_slug,
        ).first()
        if schedule:
            schedule.is_important = True
            schedule.save(update_fields=('is_important',))
        else:
            log.warning(f'No such schedule: {schedule_slug}, service: {service_slug}')

    if to_remove_important:
        log.info(f'Removing {len(to_remove_important)} is_important attribute')
        Schedule.objects.filter(pk__in=to_remove_important).update(
            is_important=False,
        )

    log.info('Finish syncing important schedules')
