import logging

from django.conf import settings
from startrek_client import Startrek, settings as startrek_settings
from yandex_tracker_client.exceptions import Forbidden
from plan.common.utils.tvm import get_tvm_ticket

logger = logging.getLogger(__name__)


def _get_tracker_client(user_ticket=None, api_version=startrek_settings.VERSION_V2):
    return Startrek(
        base_url=settings.TRACKER_API_URL,
        api_version=api_version,
        useragent=settings.ABC_IDS_USER_AGENT,
        user_ticket=user_ticket,
        service_ticket=get_tvm_ticket(destination=str(settings.TRACKER_TVM_ID))
    )


def check_can_grant_queue(queue_id, user_ticket):
    logger.info(f'Checking grant permission on {queue_id}')
    client = _get_tracker_client(user_ticket=user_ticket)
    try:
        queue = client.queues[queue_id]
    except Forbidden:
        return False
    permissions = queue.check_permissions(permission_code='grant')
    users = getattr(permissions, 'users', None)
    return bool(users)


def enable_auto_assign(component_id, user_ticket):
    logger.info(f'Enabling assignAuto of {component_id}')
    client = _get_tracker_client(user_ticket=user_ticket)
    component = client.components[component_id]
    if not component.assignAuto:
        component.update(assignAuto=True)
    logger.info(f'Enable assignAuto of {component_id}')


def set_component_lead(component_id, staff):
    logger.info(f'Setting lead of {component_id} to {staff.login}')
    client = _get_tracker_client(api_version=startrek_settings.VERSION_SERVICE)
    component = client.components[component_id]
    lead = component._value.get('lead')
    if lead:
        if lead._value['id'] == staff.login:
            logger.info(f'Skipping set lead for {component_id} - already needed lead')
            return
        logger.info(
            f'Replacing lead of {component_id} from {lead._value["id"]} to {staff.login}'
        )
    component.update(lead=staff.login)
    logger.info(f'Set lead of {component_id} to {staff.login}')


def remove_component_lead(component_id, staff=None):
    logger.info(f'Removing lead of {component_id}')
    client = _get_tracker_client(api_version=startrek_settings.VERSION_SERVICE)
    component = client.components[component_id]
    lead = component._value.get('lead')
    if lead:
        if staff and lead._value['id'] != staff.login:
            logger.info(f'Skipping remove lead for {component_id} - unexpected lead')
            return
        component.update(lead={"unset": None})
    logger.info(f'Remove lead of {component_id}')
