from django.utils.translation import ugettext_lazy as _


class Error(Exception):
    """
    Эксепшн, наследников которого нужно райзить в методах вьюх,
    наследованных от UniversalView.
    Каждый наследник Error соответствует http-статусу (403, 409, 500 и т.д.)
    Сама по себе Error рейзить не стоит.
    Не кодируем ошибки в названиях классов, а вместо этого
    инициализируем их эрроркодами.
    Можно сделать маппинг эрроркодов на сообщения и заглядывать туда, если
    сообщение не передано.

    Usage:

    class MyView(UniversalView):
        def get(self, request, *args, **kwargs):
            ...
            if not valid:
                raise СonflictError(
                    error_code='DATABASE_EXPLODED',
                )
    """
    http_code = 500
    error_code = None
    message = None
    params = None

    def __init__(self, error_code=None, message=None, params=None):
        self.error_code = error_code or self.error_code
        self.message = message or self.message
        self.params = params if params is not None else {}

    def __repr__(self):
        return '[{s.http_code}] {s.error_code}'.format(s=self)


class PermissionError(Error):
    http_code = 403
    error_code = 'PERMISSION_DENIED'
    message = 'Permission denied'


class ConflictError(Error):
    http_code = 409
    error_code = 'ERROR'
    message = 'Something bad happened'


class NotFoundError(Error):
    http_code = 404
    error_code = 'NOT_FOUND'
    message = 'Not found'


class FormConflictError(ConflictError):
    """
    Специальный случай ConflictError, когда все ошибки можно получить из формы.
    """
    def __init__(self, form):
        common_errors = form.errors.get('__all__')

        self.error_code = 'VALIDATION_ERROR'

        self.params = [form.errors]

        if common_errors:
            self.message = str(common_errors[0])
        else:
            self.message = str(_('Некорректные параметры запроса'))


class DataValidationError(Error):
    """Когда в данных запроса нет ошибок или
    конфликтов, но данные некорректны семантически.
    """
    http_code = 422
    error_code = 'DATA_INCORRECT'
    message = 'Request data is incorrect'


class HttpRedirect(Exception):
    def __init__(self, redirect_to):
        self.redirect_to = redirect_to


class ContactError(Error):
    text_message = _('Неправильно заполнена форма создания ссылки')


class IDSRepositoryError(Error):
    message = 'Не удалось подключиться к трекеру'


class QueueDoesNotExist(Error):
    message = _('Такой очереди не существует')
