import datetime
from itertools import groupby
from typing import Iterable, Dict, Tuple

from rest_framework import viewsets, status
from rest_framework.response import Response

from plan.holidays.api.serializers import HolidaySerializer
from plan.holidays.models import Holiday
from plan.swagger import SwaggerFrontend


class HolidayView(viewsets.ViewSet):
    default_swagger_schema = SwaggerFrontend
    _permissions_to_proceed = 'can_view'

    def list(self, request):
        serializer = HolidaySerializer(data=request.GET)
        if serializer.is_valid():
            date_from, date_to = (serializer.validated_data[arg] for arg in ('date_from', 'date_to'))
            results = list(self.get_holidays(date_from, date_to))
            return Response(
                {'results': results},
                status=status.HTTP_200_OK,
            )
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @staticmethod
    def serialize_date(date: datetime.date) -> str:
        return date.strftime('%Y-%m-%d')

    @staticmethod
    def group_key(row_number_and_date) -> Tuple[datetime.date, bool]:
        # https://www.python.org/dev/peps/pep-3113/
        row_number, (date, is_weekend) = row_number_and_date
        return date - datetime.timedelta(days=row_number), is_weekend

    @classmethod
    def get_holidays(cls, date_from: datetime.date, date_to: datetime.date) -> Iterable[Dict[str, str]]:
        # TODO: все это можно сделать с помощью Window и RowNumber в django >= 2.0

        sorted_holidays = (
            (obj.date, obj.is_weekend)
            for obj in Holiday.objects.filter(date__range=(date_from, date_to)).order_by('date')
        )

        groups = [
            tuple(item[1] for item in group)
            for key, group in groupby(enumerate(sorted_holidays), cls.group_key)
        ]

        ranges = (
            (group[0], group[-1])
            for group in groups
        )

        results = (
            {
                'start': cls.serialize_date(group[0][0]),
                'end': cls.serialize_date(group[1][0]),
                'interval_type': cls.get_internal_type(group[0][1])
            }
            for group in ranges
        )

        return results

    @staticmethod
    def get_internal_type(is_weekend):
        if is_weekend:
            return 'weekend'
        return 'holiday'
