import logging
from datetime import datetime, date
from typing import Iterable
from dateutil.relativedelta import relativedelta
from requests import RequestException

from lxml import etree
from django.conf import settings

from plan.common.utils.http import Session


log = logging.getLogger(__name__)


class CalendarConnectionError(Exception):
    pass


def _import_holidays(dtstart, dtend, max_retries=None):
    """
    Ходим в ручку календаря вида
    https://calendar.yandex.ru/holidays.xml?start_date=21-01-01&end_date=21-01-06&country_id=225&out_mode=all
    """

    log.warning('Request export/holidays.xml')

    url = ("%s://%s/export/holidays.xml"
           "?start_date=%s&end_date=%s&country_id=%d&out_mode=all")

    url = url % (
        settings.CALENDAR_API_PROTO,
        settings.CALENDAR_API_HOST,
        dtstart.strftime('%Y-%m-%d'),
        dtend.strftime('%Y-%m-%d'),
        225,  # Россия
    )

    session_kwargs = {
        'read_retries': max_retries or settings.DEFAULT_READ_RETRIES,
        'connect_retries': max_retries or settings.DEFAULT_CONNECT_RETRIES,
    }

    try:
        with Session(**session_kwargs) as session:
            r = session.get(url, timeout=settings.CALENDAR_IMPORT_TIMEOUT)
    except RequestException as e:
        raise CalendarConnectionError from e

    if r.status_code != 200:
        raise RuntimeError('Calendar return %s' % r.status_code)

    tree = etree.XML(r.text.encode('utf-8'))
    days = tree.find('get-holidays/days')
    # FutureWarning: The behavior of this method will change in future
    # versions. Use specific 'len(elem)' or 'elem is not None' test instead.
    # TODO: use ids.calendar
    if days:
        for e in days.iterchildren():
            yield e.attrib


def _is_holiday(day):
    return day.get('day-type') == 'holiday'


def _is_not_workday(day):
    return day.get('is-holiday') == '1' or day.get('day-type') == 'weekend'


def get_remote_holidays(date_from: datetime.date, date_to: datetime.date) -> Iterable[datetime.date]:
    raw_daydata = _import_holidays(date_from, date_to)
    return {
        datetime.strptime(day['date'], '%Y-%m-%d').date(): _is_holiday(day)
        for day in raw_daydata
        if _is_not_workday(day)
    }


def _russian_days_of_month(year, month, is_holiday, month_count=0):
    dtstart = date(year, month, 1)
    dtend = dtstart + relativedelta(months=+1+month_count, days=-1)

    for day in _import_holidays(dtstart, dtend):
        if is_holiday == _is_not_workday(day):
            dt = datetime.strptime(day['date'], '%Y-%m-%d').date()
            yield dt


def days_of_month(year, month):
    dtstart = date(year, month, 1)
    dtend = dtstart + relativedelta(months=+1, days=-1)
    return [date(year, month, i) for i in range(1, dtend.day + 1)]


def workdays_of_month(year, month):
    return list(_russian_days_of_month(year, month, is_holiday=False))


def holidays_of_month(year, month):
    return list(_russian_days_of_month(year, month, is_holiday=True))


def day_is_workdays(day, list_workdays=None):
    if list_workdays is None:
        return day in workdays_of_month(day.year, day.month)

    return day in list_workdays


def get_list_workdays(day, month_count):
    # month_count - число следующих месяцев, информацию о рабочих днях которых нужно подгрузить к текущему
    is_holiday = False
    return list(_russian_days_of_month(day.year, day.month, is_holiday, month_count))


def next_workday(day, list_workdays=None):
    if list_workdays is None:
        list_workdays = get_list_workdays(day, 2)

    for next_day in list_workdays:
        if next_day > day:
            return next_day
