import datetime
import logging
from typing import Iterable, Union

from django.db import models
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from plan.holidays.calendar import get_remote_holidays


log = logging.getLogger(__name__)


class Holiday(models.Model):
    date = models.DateField(primary_key=True)
    is_holiday = models.BooleanField(default=False, verbose_name=_('Является праздником'))

    class Meta:
        ordering = ['date']

    def __str__(self):
        return str(self.date)

    @classmethod
    def sync(cls, date_from: datetime.date, date_to: datetime.date):
        new_holidays = get_remote_holidays(date_from, date_to)
        if new_holidays is None:
            log.warning(f'Got empty data: {date_from} - {date_to}')

        new_holidays_date = set(new_holidays.keys())
        old_holidays = cls.objects.filter(date__range=(date_from, date_to))
        date_for_delete = set()
        for holiday_obj in old_holidays:
            if holiday_obj.date not in new_holidays or holiday_obj.is_holiday != new_holidays[holiday_obj.date]:
                log.info(f'Deleting holiday: {holiday_obj.date}')
                date_for_delete.add(holiday_obj.date)

        if date_for_delete:
            old_holidays.filter(date__in=date_for_delete).delete()

        new_objects = [
            Holiday(date=date, is_holiday=new_holidays[date])
            for date in new_holidays_date.difference(
                old_holidays.exclude(date__in=date_for_delete).values_list('date', flat=True)
            )
        ]
        if new_objects:
            cls.objects.bulk_create(new_objects)

    @classmethod
    def not_workday(cls, date: datetime.date) -> bool:
        """
        Этот метод должен использоваться только в местах, где нет настроек праздники/выходные.
        Например, дайджест светофора.
        Здесь учитываем любой нерабочий день, без градации просто выходной или праздничный.
        """
        return cls.objects.filter(date=date).exists()

    @property
    def is_weekend(self):
        return not self.is_holiday

    def is_workday(self, duty_on_holidays: bool, duty_on_weekends: bool) -> bool:
        """
        Возвращает признак является ли праздник рабочим в зависимости от настроек графика.
        """
        # если в настройках дежурим и в выхи, и праздники - всегда рабочий
        if duty_on_holidays and duty_on_weekends:
            return True

        # выходной или праздник с соотвествующими настройками
        return (self.is_weekend and duty_on_weekends) or (self.is_holiday and duty_on_holidays)

    @classmethod
    def date_obj(cls, date: datetime.date) -> Union['Holiday', None]:
        try:
            date = cls.objects.get(date=date)
            return date
        except cls.DoesNotExist:
            return None

    @classmethod
    def date_is_holiday_type(cls, date: datetime.date) -> Union[bool, None]:
        """
        Троичная система для типа объекта Holiday:
            * True - праздник
            * False - выходной
            * None - нет объекта, тк это рабочий
        """
        date_obj = cls.date_obj(date)
        if date_obj is not None:
            return date_obj.is_holiday

    @classmethod
    def workday(cls, date: datetime.date, duty_on_holidays: bool, duty_on_weekends: bool) -> bool:
        holiday = cls.date_obj(date)
        if holiday is not None:
            return holiday.is_workday(duty_on_holidays, duty_on_weekends)

        return True

    @classmethod
    def next_workday(cls, date: datetime.date, duty_on_holidays: bool, duty_on_weekends: bool) -> (datetime.date, datetime.timedelta):
        delta = 0
        while not cls.workday(date, duty_on_holidays, duty_on_weekends):
            date += timezone.timedelta(days=1)
            delta += 1

        return date, timezone.timedelta(days=delta)

    @classmethod
    def prev_workday(cls, date: datetime.date, duty_on_holidays: bool, duty_on_weekends: bool) -> (datetime.date, datetime.timedelta):
        delta = -1
        date -= timezone.timedelta(days=1)
        while not cls.workday(date, duty_on_holidays, duty_on_weekends):
            date -= timezone.timedelta(days=1)
            delta -= 1

        return date, timezone.timedelta(days=delta)

    @classmethod
    def holidays_range(cls, date_from: datetime.date, date_to: datetime.date) -> Iterable[datetime.date]:
        """inclusive"""
        yield from (holiday.date for holiday in cls.objects.filter(date__range=(date_from, date_to)))
