import itertools

from django.conf import settings

from plan.common.utils.intranet import superiors
from plan.notify.recipients import objects as recipient_objects
from plan.services import models as service_models


class DisputeBase(object):
    @property
    def notification_id(self):
        raise NotImplementedError

    def __init__(self, request_person, person, role, service, comment=None):
        self.request_person = request_person
        self.person = person
        self.role = role
        self.service = service
        self.comment = comment

    def check_permission(self):
        raise NotImplementedError

    def get_recipients(self):
        raise NotImplementedError

    def get_cc_recipients(self):
        return {recipient_objects.Recipient(
            settings.OCCUPANCY_PARTICIPATION_DISPUTE_EMAIL)}

    def get_context(self):
        return {
            'service': self.service,
            'person': self.person,
            'role': self.role,
            'comment': self.comment,
            'request_person': self.request_person,
            'footer_mode': 'occupancy',
        }


class ReviewDispute(DisputeBase):
    notification_id = 'notifications.hr.person_participation_dispute_review'

    def check_permission(self):
        return self.request_person.staff == self.person

    def get_recipients(self):
        recipients = set()

        superior = superiors.get_superior_of_person(self.person)
        if superior:
            recipients.add(superior)

        if self.service.owner:
            recipients.add(self.service.owner)

        return recipients

    def get_cc_recipients(self):
        cc = super(ReviewDispute, self).get_cc_recipients()
        cc.add(self.person),
        return cc


class RaiseDispute(DisputeBase):
    notification_id = 'notifications.hr.person_participation_dispute_raise'

    _service_owners = None
    _conflicting_memberships = None

    def check_permission(self):
        return self.request_person.is_head(self.service)

    def get_conflicting_memberships(self):
        if self._conflicting_memberships is None:
            members_qs = service_models.ServiceMember.objects.filter(
                staff=self.person,
                service__state__in=service_models.Service.states.ALIVE_STATES,
                part_rate__gt=0,
            )

            self._conflicting_memberships = list(
                members_qs.select_related('service').order_by('service__id'))

        return self._conflicting_memberships

    def get_recipients(self):
        recipients = set()

        superior = superiors.get_superior_of_person(self.person)
        if superior:
            recipients.add(superior)

        memberships = self.get_conflicting_memberships()
        affected_services = {m.service for m in memberships}

        for service in affected_services:
            if service.owner:
                recipients.add(service.owner)

        if self.service.owner:
            recipients.add(self.service.owner)

        return recipients

    def get_cc_recipients(self):
        cc = super(RaiseDispute, self).get_cc_recipients()
        cc.add(self.request_person.staff),
        return cc

    def get_context(self):
        context = super(RaiseDispute, self).get_context()
        members = self.get_conflicting_memberships()
        grouped_members = itertools.groupby(members, key=lambda m: m.service)

        context['conflicting_services'] = []
        for service, membershpis in grouped_members:
            service_doc = {
                'slug': service.slug,
                'name': service.name,
                'memberships': []
            }

            for member in membershpis:
                if all((member.staff == self.person, member.service == self.service, member.role == self.role)):
                    continue

                service_doc['memberships'].append(member)

            if service_doc['memberships']:
                context['conflicting_services'].append(service_doc)

        return context


DISPUTE_BY_TYPE = {
    'review': ReviewDispute,
    'raise': RaiseDispute,
}
