import collections

from django import forms

from plan.common.utils import forms as utils_forms
from plan.roles import models as role_models
from plan.services import models as service_models
from plan.staff.models import Staff


class ImportantServicesForm(forms.Form):
    year_from = forms.IntegerField(min_value=2010, max_value=3000)
    month_from = forms.IntegerField(min_value=1, max_value=12)
    year_to = forms.IntegerField(min_value=2010, max_value=3000)
    month_to = forms.IntegerField(min_value=1, max_value=12)


class SubordinatesForm(utils_forms.DefaultDataMixin, forms.Form):
    default_data = {'direct': False}

    direct = forms.BooleanField(required=False)


class PersonParticipationDisputeForm(forms.Form):
    service = forms.ModelChoiceField(
        queryset=service_models.Service.objects.filter(
            state__in=service_models.Service.states.ALIVE_STATES,
        ).select_related('owner'),
    )

    role = forms.ModelChoiceField(role_models.Role.objects.all())
    comment = forms.CharField(required=False)

    def __init__(self, data, person, *args, **kwargs):
        super_self = super(PersonParticipationDisputeForm, self)
        super_self.__init__(data, *args, **kwargs)
        self.person = person

    def clean(self):
        service = self.cleaned_data.get('service')
        role = self.cleaned_data.get('role')

        if service and role:
            queryset = service.members.filter(
                staff=self.person,
                role=role,
            )

            if not queryset.exists():
                raise forms.ValidationError(
                    "Invalid combination of service, person and role.")

        return self.cleaned_data


RateChangeData = collections.namedtuple(
    'RateChangeData', 'person service role rate')


class ParticipationRateChangeForm(forms.Form):
    login = forms.ModelChoiceField(
        queryset=Staff.objects.filter(is_dismissed=False),
        to_field_name='login',
    )
    service_id = forms.ModelChoiceField(
        service_models.Service.objects.filter(
            state__in=service_models.Service.states.ALIVE_STATES)
    )
    role_id = forms.ModelChoiceField(role_models.Role.objects.all())
    rate = forms.DecimalField(max_digits=5, decimal_places=4)

    def get_change_data(self):
        if self.is_valid():
            return RateChangeData(
                person=self.cleaned_data['login'],
                service=self.cleaned_data['service_id'],
                role=self.cleaned_data['role_id'],
                rate=self.cleaned_data['rate'],
            )
