import io
import calendar
import collections
import datetime
import decimal
from typing import Dict, List


import pytz
from django.conf import settings
from django.core import urlresolvers
from django.db import models as db_models
from django.db.models import Q, Subquery, Max
from django.utils import timezone

from plan import exceptions
from plan.api.intranet.persons import (
    LegacyPersonSerializer,
    LegacyPersonSerializerWithDepartment,
    LegacyPersonWithSubordinates,
)
from plan.common.person import Person
from plan.common.utils import xls
from plan.hr import models as hr_models
from plan.roles.models import Role
from plan.services import models as services_models
from plan.services.api.team import RoleSerializer
from plan.services.permissions import is_service_responsible_bulk
from plan.services.models import Service
from plan.services.views.catalog.serializers import ServiceLiteSerializer
from plan.staff.models import Staff

DECIMAL_ZERO = decimal.Decimal(0)

# 25 число каждого месяца день когда hr подбивают все итоги
HR_REPORTING_DAY = 25

person_serializer = LegacyPersonSerializer
person_serializer_with_department = LegacyPersonSerializerWithDepartment
person_serializer_with_subordinates = LegacyPersonWithSubordinates
role_serializer = RoleSerializer
service_lite_serializer = ServiceLiteSerializer


def get_service_participation_rates(service: Service, request_person: Person) -> List[Dict[str, Dict]]:
    aggr = collections.defaultdict(list)
    service_members = services_models.ServiceMember.objects.filter(
        service=service,
        staff__is_dismissed=False,
    ).select_related('staff', 'role', 'staff__department')

    is_hr = request_person.user.has_perm('internal_roles.change_importance')
    is_responsible = request_person.is_responsible(service)
    subordinate_logins = request_person.staff.get_subordinate_logins(only_direct=False)

    persons = {member.staff for member in service_members}
    total_rate_by_login = get_persons_total_occupancy(persons)

    for sm in service_members:
        scope_name = sm.role.scope.name
        is_subordinate = sm.staff.login in subordinate_logins

        aggr[scope_name].append({
            'permissions': {
                'can_edit_all_participation_rates': is_subordinate or is_hr,
                'can_edit_participation_rates': is_hr or is_responsible or is_subordinate,
            },
            'person': person_serializer_with_department(sm.staff).data,
            'role': role_serializer(sm.role).data,
            'rate': sm.part_rate,
            'total_rate': total_rate_by_login.get(sm.staff.login) or DECIMAL_ZERO,
        })

    result = [
        {'scope': k, 'persons': v}
        for k, v in aggr.items()
    ]
    result.sort(key=lambda o: o['scope'])
    for r in result:
        r['persons'].sort(key=lambda o: o['person']['full_name'])
    return result


def get_persons_participation_rates(person_qs,
                                    is_superior_or_hr: bool = False,
                                    requester: Staff = None,
                                    chief: Staff = None) -> Dict[str, Dict]:
    partrate_queryset = services_models.ServiceMember.objects.filter(
        staff__in=person_qs,
        service__state__in=services_models.Service.states.ALIVE_STATES,
    ).select_related(
        'service',
        'role'
    ).order_by('staff_id').values(
        'staff__login',
        'staff__chief_id',
        'service_id',
        'role_id',
        'part_rate'
    )

    partrate_data = list(partrate_queryset)
    service_ids = {item['service_id'] for item in partrate_data}
    role_ids = {item['role_id'] for item in partrate_data}

    services = Service.objects.filter(pk__in=service_ids).select_related('owner')
    roles = Role.objects.filter(pk__in=role_ids).select_related('service')

    serialized_persons = {
        staff.login: person_serializer_with_department(staff).data for staff in person_qs.select_related('department')
    }
    serialized_services = {
        service.id: service_lite_serializer(service).data for service in services
    }
    serialized_roles = {
        role.id: role_serializer(role).data for role in roles
    }

    persons = {}
    permission_data = {}
    if requester is not None:
        permissions = is_service_responsible_bulk(services, Person(requester))
        for service, has_perm in permissions.items():
            permission_data[service.id] = has_perm

    for item in partrate_data:
        login: str = item['staff__login']
        chief_id = item['staff__chief_id']
        service_id = item['service_id']
        role_id = item['role_id']
        part_rate = item['part_rate']
        if login not in persons:
            persons[login] = {}
            if chief is not None:
                persons[login] = {
                    'is_direct': chief_id == chief.id
                }

            persons[login]['participation_data'] = []
            persons[login]['person'] = serialized_persons[login]

        participation_data = persons[login]['participation_data']
        participation_item = {
            'service': serialized_services[service_id],
            'role': serialized_roles[role_id],
            'rate': part_rate,
        }
        if requester is not None:
            is_responsible = permission_data.get(service_id)
            can_edit = is_superior_or_hr or is_responsible
            participation_item['permissions'] = {
                'can_edit_service_participation_rates': can_edit
            }
        participation_data.append(participation_item)

    for part_data in persons.values():
        part_data['participation_data'].sort(
            key=lambda item: (item['service']['name'], item['rate'] or 0),
        )

    return persons


def get_persons_total_occupancy(persons) -> Dict[str, decimal.Decimal]:
    queryset = Staff.objects.filter(
        id__in=[p.id for p in persons],
        servicemembers__service__state__in=services_models.Service.states.ALIVE_STATES,
    )

    annotation = queryset.annotate(
        total_rate=db_models.Sum('servicemembers__part_rate')
    )

    return {person.login: person.total_rate for person in annotation}


def get_service_participation_history(service):
    qs = hr_models.PartRateHistory.objects.filter(
        service_member__service=service
    )
    return _compose_history(qs)


def get_person_participation_history(person):
    threshold = timezone.now() - datetime.timedelta(days=90)
    qs = hr_models.PartRateHistory.objects.filter(
        service_member__staff=person,
        modified_at__gte=threshold,
    )
    return _compose_history(qs)


def _compose_history(query_set):
    query_set = query_set.select_related(
        'service_member__staff__department',
        'service_member__service__owner',
        'staff',
        'staff__department',
    ).order_by('-modified_at', '-id')
    result = []
    for obj in query_set:
        result.append({
            'person': person_serializer_with_department(obj.service_member.staff).data,
            'person_changed': person_serializer_with_department(obj.staff).data,
            'service': service_lite_serializer(obj.service_member.service).data,
            'timestamp': obj.modified_at,
            'old': obj.old_part_rate,
            'new': obj.new_part_rate,
            'role': role_serializer(obj.service_member.role).data,
        })
    return result


def get_person_subordinates_history(person, only_direct=True):
    subordinates_qs = person.get_subordinate_staffs(only_direct)
    if not subordinates_qs.exists():
        return []

    threshold = timezone.now() - datetime.timedelta(days=90)
    qs = hr_models.PartRateHistory.objects.filter(
        service_member__staff__in=list(subordinates_qs),
        modified_at__gte=threshold,
    )
    return _compose_history(qs)


def set_persons_rates(changes, request_person):
    """Проверяет и применяет изменения коэфицентов участия.
    Если сумма новых коэфицентов участия не больше 1 и
    если в данных на изменение нет несуществующих в АБЦ участий, то
    новые коэфиценты применяются, иначе исключение.
    """
    changes_dict = collections.defaultdict(dict)
    for change in changes:
        subkey = (change.service.id, change.role.id)
        changes_dict[change.person.login][subkey] = change.rate

    service_members_qs = services_models.ServiceMember.objects.filter(
        staff__login__in=list(changes_dict.keys()),
        service__state__in=services_models.Service.states.ALIVE_STATES,
    )
    # person -> {service_member: rate, ...}
    current_dict = collections.defaultdict(dict)
    update_dict = collections.defaultdict(dict)
    for sm in service_members_qs:
        person = sm.staff
        current_dict[person][sm] = sm.part_rate

        t = (sm.service.id, sm.role.id)
        if t in changes_dict[person.login]:
            new_rate = changes_dict[person.login][t]
            new_rate = decimal.Context(prec=5).create_decimal(new_rate)
            # если коэфицент действительно изменился
            if new_rate != sm.part_rate:
                update_dict[person][sm] = new_rate
            del changes_dict[person.login][t]

    # тут не должно остаться никаких данных
    if any(changes_dict.values()):
        raise exceptions.DataValidationError()

    for (person, update_data) in update_dict.items():
        current_data = current_dict[person]
        new_data = current_data.copy()
        new_data.update(update_data)
        if not _validate_new_rates(list(new_data.values())):
            raise exceptions.DataValidationError()

        for (sm, new_rate) in update_data.items():
            create_history_object(
                sm,
                request_person.staff,
                sm.part_rate,
                new_rate,
            )
            sm.part_rate = new_rate
            sm.save()


def _validate_new_rates(rates):
    return sum([_f for _f in rates if _f]) <= 1


def create_history_object(service_member, staff, old, new):
    hist_obj = hr_models.PartRateHistory(
        service_member=service_member,
        member_staff=service_member.staff,
        staff=staff,
        old_part_rate=old,
        new_part_rate=new,
    )
    hist_obj.save()


def get_team_xls(start, end):
    start = _to_datetime(start)
    end = _to_datetime(end)
    # границу включаем
    end = _add_month(end)
    history = _fetch_history(start, end)
    history_dict = collections.defaultdict(list)
    for h_obj in history:
        sm = h_obj.service_member
        history_dict[sm].append(h_obj)

    buffer = io.BytesIO()
    header = [
        'Логин',
        'ФИО',
        'Сервис',
        'URL сервиса',
        'Роль',
        'Главность',
        'Статус',
    ] + ['КУ (%s)' % mn for mn in _month_names(start, end)]
    rows = [header] + [
        _member2row(member, hist, start, end)
        for (member, hist) in history_dict.items()
    ]
    xls.write_rows_xls(rows, out=buffer)
    return buffer.getvalue()


def _fetch_history(start, end):
    important_history = hr_models.PartRateHistory.objects.filter(
        service_member__service__is_important=True,
        service_member__service__state__in=services_models.Service.states.ALIVE_STATES,
        service_member__staff__is_dismissed=False,
    )

    return important_history.filter(
        Q(modified_at__gte=start, modified_at__lte=end) |
        Q(pk__in=Subquery(
            important_history
            .filter(modified_at__lt=start)
            .values('service_member_id')
            .annotate(max_id=Max('id'))
            .values_list('max_id', flat=True)
        ))
    ).select_related(
        'service_member'
    ).order_by('modified_at', 'id')


def _month_names(start, end):
    month_names = {
        1: 'Январь',
        2: 'Февраль',
        3: 'Март',
        4: 'Апрель',
        5: 'Май',
        6: 'Июнь',
        7: 'Июль',
        8: 'Август',
        9: 'Сентябрь',
        10: 'Октябрь',
        11: 'Ноябрь',
        12: 'Декабрь',
    }
    result = []
    current_datetime = start
    while current_datetime < end:
        current_month = current_datetime.month
        month_name = month_names[current_month]
        result.append(month_name)
        current_datetime = _add_month(current_datetime)
    return result


def _member2row(member, history, start, end):
    assert len(history) > 0, (
        'Invalid history data for member: %s' % member.id
    )

    service_url = urlresolvers.reverse(
        'services:service', args=[member.service_id]
    )

    is_senior = Person(member.staff).is_responsible(member.service)
    row = [
        member.staff.login,
        member.staff.get_full_name(),
        member.service.name,
        settings.ABC_URL + service_url,
        member.role.name,
        'главный' if is_senior else 'не главный',
        'подтверждён',
    ]

    reporting_day = start.replace(day=HR_REPORTING_DAY)
    index = 0
    while reporting_day < end:
        while index + 1 < len(history) and history[index + 1].modified_at <= reporting_day:
            index += 1
        if history[index].modified_at > reporting_day:
            rate_for_rday = '-'
        elif history[index].new_part_rate:
            rate_for_rday = history[index].new_part_rate
        else:
            rate_for_rday = DECIMAL_ZERO
        row.append(str(rate_for_rday))
        reporting_day = _add_month(reporting_day)

    return row


def _add_month(sourcedate, delta=1):
    month = sourcedate.month - 1 + delta
    year = int(sourcedate.year + month / 12)
    month = month % 12 + 1
    day = min(sourcedate.day, calendar.monthrange(year, month)[1])
    return _to_datetime(datetime.date(year, month, day))


def _to_datetime(date_value):
    tz = pytz.timezone(settings.TIME_ZONE)
    return datetime.datetime(
        date_value.year,
        date_value.month,
        date_value.day,
        tzinfo=tz,
    )
