import datetime
import json

from django import http
from django.contrib.auth import decorators as auth_decorators
from django.conf import settings
from django.db import transaction
from django.utils import decorators

from rest_framework.views import APIView
from rest_framework.permissions import IsAuthenticated

from plan import exceptions
from plan.api.intranet.persons import (
    LegacyPersonSerializer,
    LegacyPersonWithSubordinates,
)
from plan.common.utils.intranet import superiors
from plan.common.utils.rest_fw import plan_response
from plan.common.models import str_from_date
from plan.hr import disputes
from plan.hr import forms
from plan.hr import helpers
from plan.hr import tasks
from plan.lib.views import base
from plan.services import models as services_models
from plan.services.views.catalog.serializers import ServiceLiteSerializer
from plan.notify.shortcuts import deliver_email
from plan.staff.models import Staff


class ImportantServiceTeamsView(base.UniversalView):

    @decorators.method_decorator(
        auth_decorators.permission_required('internal_roles.change_importance')
    )
    def dispatch(self, *args, **kwargs):
        return super(ImportantServiceTeamsView, self).dispatch(*args, **kwargs)

    def get(self, request):
        form = forms.ImportantServicesForm(request.GET)
        if not form.is_valid():
            raise exceptions.FormConflictError(form)
        month_start = datetime.date(
            year=form.cleaned_data['year_from'],
            month=form.cleaned_data['month_from'],
            day=1,
        )
        month_end = datetime.date(
            year=form.cleaned_data['year_to'],
            month=form.cleaned_data['month_to'],
            day=1,
        )
        if month_start > month_end:
            month_end, month_start = month_start, month_end

        tasks.make_team_xls.delay(request.user.username, str_from_date(month_start), str_from_date(month_end))
        return {'message': 'Запущена генерация отчета. Файл отчета придет вам на почту.'}


class MarkImportantView(base.UniversalView):
    @decorators.method_decorator(
        auth_decorators.permission_required('internal_roles.change_importance')
    )
    def dispatch(self, *args, **kwargs):
        return super(MarkImportantView, self).dispatch(*args, **kwargs)

    def post(self, request):
        services = self.json_data
        for sid in services:
            try:
                service = services_models.Service.objects.get(id=sid)
            except services_models.Service.DoesNotExists:
                raise http.Http404
            service.is_important = True
            service.save(update_fields=('is_important',))
        return {}


class ServiceParticipationView(APIView):
    permission_classes = (IsAuthenticated, )

    def get(self, request, pk_or_slug):
        try:
            service = services_models.Service.objects.alive().select_related('owner').get_by_pk_or_slug(pk_or_slug)
        except services_models.Service.DoesNotExist:
            raise http.Http404

        request_person = request.person
        return plan_response({
            'service': ServiceLiteSerializer(service).data,
            'participation_data': helpers.get_service_participation_rates(service, request_person),
            'permissions': {
                'can_send_dispute_request': request_person.is_head(service),
            },
        })


class ServiceParticipationHistoryView(APIView):
    permission_classes = (IsAuthenticated, )

    def get(self, request, pk_or_slug):
        try:
            service = services_models.Service.objects.alive().select_related('owner').get_by_pk_or_slug(pk_or_slug)
        except services_models.Service.DoesNotExist:
            raise http.Http404
        history = helpers.get_service_participation_history
        return plan_response({
            'service': ServiceLiteSerializer(service).data,
            'participation_history': history(service),
        })


class PersonParticipationView(APIView):
    permission_classes = (IsAuthenticated, )

    def get(self, request, login):
        staff_qs = Staff.objects.select_related('department', 'chief').filter(login=login)
        try:
            staff = staff_qs.get()
        except Staff.DoesNotExist:
            raise http.Http404
        person_doc = LegacyPersonWithSubordinates(staff).data

        request_person = request.person
        is_superior = superiors.is_superior_of_person(request_person.staff, staff)
        is_hr = request_person.user.has_perm('internal_roles.change_importance')
        is_superior_or_hr = is_superior or is_hr
        rates = helpers.get_persons_participation_rates(staff_qs, is_superior_or_hr, request.person.staff, staff.chief)
        participation_data = rates.get(staff.login, {}).get('participation_data', [])

        return plan_response({
            'person': person_doc,
            'participation_data': participation_data,
            'permissions': {
                'can_send_dispute_request': request_person.staff == staff,
                'can_edit_participation_rates': is_superior_or_hr,
            }
        })


class PersonParticipationDisputeView(APIView):
    permission_classes = (IsAuthenticated, )

    def dispatch(self, request, login, mode, *args, **kwargs):
        try:
            person = Staff.objects.select_related('department').get(login=login)
        except Staff.DoesNotExist:
            raise http.Http404

        if request.method == 'POST':
            try:
                data = json.load(request)
            except ValueError:
                raise exceptions.Error(
                    error_code='JSON_DECODE_ERROR',
                    message='No JSON object could be decoded',
                )
        else:
            data = request.GET

        form = forms.PersonParticipationDisputeForm(data, person)
        if not form.is_valid():
            return plan_response(
                error=exceptions.FormConflictError(form),
            )

        dispute_cls = disputes.DISPUTE_BY_TYPE[mode]
        dispute = dispute_cls(
            request_person=request.person,
            person=person,
            role=form.cleaned_data['role'],
            service=form.cleaned_data['service'],
            comment=form.cleaned_data['comment'],
        )

        if not dispute.check_permission():
            return plan_response(error=exceptions.PermissionError())

        super_self = super(PersonParticipationDisputeView, self)
        return super_self.dispatch(request, dispute)

    def get(self, request, dispute):
        visible_recipients = dispute.get_recipients()
        return plan_response({
            'recipients': [LegacyPersonSerializer(recipient).data for recipient in visible_recipients],
            'copy': settings.OCCUPANCY_PARTICIPATION_DISPUTE_EMAIL,
        })

    def post(self, request, dispute):
        recipients = dispute.get_recipients()
        deliver_email(
            notification_id=dispute.notification_id,
            from_email=settings.OCCUPANCY_FROM_EMAIL,
            recipients=[recipients],
            cc=dispute.get_cc_recipients() - recipients,
            reply_to=settings.OCCUPANCY_PARTICIPATION_DISPUTE_EMAIL,
            context=dispute.get_context(),
        )

        return plan_response({})


class PersonParticipationSubordinatesView(APIView):
    permission_classes = (IsAuthenticated, )

    def get(self, request, login):
        form = forms.SubordinatesForm(request.GET)
        if not form.is_valid():
            raise exceptions.FormConflictError(form)
        is_direct = form.cleaned_data['direct']
        try:
            staff = Staff.objects.select_related('department').get(login=login)
        except Staff.DoesNotExist:
            raise http.Http404
        subordinates_qs = staff.get_subordinate_staffs(only_direct=is_direct)
        is_chief = subordinates_qs.exists()

        is_superior = superiors.is_superior_of_person(request.person.staff, staff)
        is_hr = request.person.user.has_perm('internal_roles.change_importance')
        part_rates = helpers.get_persons_participation_rates(
            subordinates_qs,
            is_superior or is_hr,
            request.person.staff,
            staff,
        )
        persons_data = list(part_rates.values())
        person_doc = LegacyPersonWithSubordinates(staff).data
        person_doc['is_chief'] = is_chief
        # разрешаем редактировать коэфиценты участия либо
        # hr-аналитику либо руководителю разрешаем редактировать
        # только своих подчиненных
        edit_perm = is_hr or is_superior
        return plan_response({
            'person': person_doc,
            'permissions': {
                'can_edit_participation_rates': edit_perm,
            },
            'subordinates_participation_data': persons_data,
        })


class PersonParticipationHistoryView(APIView):
    permission_classes = (IsAuthenticated, )

    def get(self, request, login):
        try:
            person = Staff.objects.select_related('department').get(login=login)
        except Staff.DoesNotExist:
            raise http.Http404
        person_doc = LegacyPersonWithSubordinates(person).data
        history = helpers.get_person_participation_history
        return plan_response({
            'person': person_doc,
            'participation_history': history(person),
        })


class PersonSubordinatesHistoryView(APIView):
    permission_classes = (IsAuthenticated, )

    def get(self, request, login):
        form = forms.SubordinatesForm(request.GET)
        if not form.is_valid():
            raise exceptions.FormConflictError(form)

        is_direct = form.cleaned_data['direct']

        try:
            staff = Staff.objects.select_related('department').get(login=login)
        except Staff.DoesNotExist:
            raise http.Http404

        person_doc = LegacyPersonWithSubordinates(staff).data
        history = helpers.get_person_subordinates_history

        return plan_response({
            'person': person_doc,
            'participation_history': history(staff, only_direct=is_direct),
        })


class EditParticipationView(base.UniversalView):
    _subordinate_logins = None

    def is_subordinate(self, person, chief):
        if self._subordinate_logins is None:
            self._subordinate_logins = chief.get_subordinate_logins(only_direct=False)
        return person.login in self._subordinate_logins

    @transaction.atomic
    def dispatch(self, *args, **kwargs):
        return super(EditParticipationView, self).dispatch(*args, **kwargs)

    def post(self, request):
        raw_changes = self.json_data
        if not isinstance(raw_changes, list):
            raise exceptions.ConflictError(message='list required')

        if not raw_changes:
            return []

        request_person = request.person
        is_hr = request_person.user.has_perm('internal_roles.change_importance')

        changes = []
        for raw_change in raw_changes:
            form = forms.ParticipationRateChangeForm(raw_change)
            if not form.is_valid():
                raise exceptions.FormConflictError(form)

            change = form.get_change_data()
            if (is_hr or request_person.is_head(change.service) or
                    self.is_subordinate(change.person, request_person.staff)):
                changes.append(change)
            else:
                raise exceptions.PermissionError()

        helpers.set_persons_rates(changes, request_person)
        return raw_changes
