import logging
from django.conf import settings
from plan.idm.exceptions import Forbidden

from plan.api import exceptions
from plan.idm.adapters.base import BaseAPI

log = logging.getLogger(__name__)


class RoleManager(BaseAPI):

    @classmethod
    def get_roles(cls, manager, filters=None, full=False, no_meta=True, light=False, offset=None):
        # GET https://idm-api.yandex-team.ru/api/v1/roles/
        log.info('Get roles with %r', filters)

        if not filters:
            filters = {}
        if 'limit' not in filters:
            filters['limit'] = settings.ABC_IDM_LIMIT

        one_page = False
        if offset is not None:
            one_page = True
            filters['offset'] = offset
        elif 'offset' not in filters:
            filters['offset'] = 0

        roles_data = {
            'objects': []
        }
        while True:
            data = manager.get('roles/', params=filters)
            if not data['objects']:
                break

            roles_data['objects'].extend(data['objects'])
            filters['offset'] += filters['limit']
            if no_meta:
                filters['no_meta'] = no_meta
            if light:
                filters['light'] = light
            if not no_meta:
                roles_data['total_count'] = data['meta']['total_count']
                roles_data['next_offset'] = filters['offset']

            if one_page:
                break
        if full:
            # expand roles data
            ids = [role['id'] for role in roles_data['objects']]

            requests = [
                {
                    'method': 'GET',
                    'path': 'roles/%s/' % role_id,
                    'body': {'_requester': filters['_requester']} if '_requester' in filters else {},
                }
                for role_id in ids
            ]

            batch_result = manager.batch(requests)

            if batch_result.ok:
                roles_data['objects'] = batch_result.successful
            else:
                raise exceptions.IntegrationError(extra=batch_result.failed)

        return roles_data

    @classmethod
    def get_role(cls, manager, pk, **params):
        # GET https://idm-api.yandex-team.ru/api/v1/roles/<id>/
        log.info('Get role with pk=%s', pk)

        role_data = manager.get('roles/%s/' % pk, params=params)
        return role_data

    @classmethod
    def request(cls, manager, pk, **kwargs):
        # POST https://idm-api.yandex-team.ru/api/v1/roles/<id>/
        log.info('Request role with pk=%s', pk)

        kwargs['state'] = 'requested'
        return manager.post('roles/%s/' % pk, data=kwargs)

    @classmethod
    def rerequest(cls, manager, pk, **kwargs):
        # POST https://idm-api.yandex-team.ru/api/v1/roles/<id>/
        log.info('Rerequest role with pk=%s', pk)

        kwargs['state'] = 'rerequested'
        return manager.post('roles/%s/' % pk, data=kwargs)

    @classmethod
    def deprive(cls, manager, pk, **kwargs):
        # DELETE https://idm-api.yandex-team.ru/api/v1/roles/<path>
        log.info('Delete role with pk=%s', pk)
        try:
            manager.delete('roles/%s/' % pk, data=kwargs)
        except Forbidden:
            role = cls.get_role(manager, pk, **kwargs)
            if not role['is_active'] or role['state'] in ('depriving', 'depriving_validation'):
                return

            raise
