import attr
import json
import logging
from urllib.parse import urljoin
from typing import Optional
import requests

from django.conf import settings

from plan.common.utils.http import Session
from plan.common.utils.tvm import get_tvm_ticket
from plan.idm import exceptions

ALREADY_HAS_ROLE_TOKEN = 'уже есть такая роль'


@attr.s
class BatchRequest(object):
    # тут довольно странное апи attr.s - валидатор optional делает сам атрибут необязательным
    # id проставляется непосредственно перед запросом в idm
    # он нужен, чтоб связать ответы idm в batch-ручке с оригинальными запросами
    # т.к. idm сам запрос вместе с его результатом не возвращает
    id = attr.ib(validator=attr.validators.optional(attr.validators.instance_of(int)))

    method = attr.ib(
        validator=attr.validators.in_(['GET', 'POST', 'PUT', 'DELETE'])
    )
    body = attr.ib()
    path = attr.ib(
        converter=lambda value: value if value.startswith('/') else '/' + value
    )


@attr.s(frozen=True)
class BatchResult(object):
    ok = attr.ib()
    successful = attr.ib()
    failed = attr.ib()


class Manager(object):
    '''
    Менеджер/адаптер сессий запросов к idm.
    Делает всю закулисную работу, нужную для авторизации,
    обрабатывает типовые ошибки от idm и дает
    делать batch-запросы через интерфейс питонячьего контекст-менеджера.
    '''
    def __init__(
        self, end_point,
        timeout: Optional[int] = None,
        retry: Optional[int] = settings.ABC_IDM_MAX_RETRY,
    ):
        self.end_point = end_point
        self.log = logging.getLogger(__name__)

        self.timeout = timeout
        self.collect_requests = False
        self.prepared_requests = {}
        self.batch_result = None
        self.retry = retry
        self._requests = None

        self.session = Session(
            connect_retries=settings.IDM_CONNECT_RETRIES,
            read_retries=retry,
        )

    def _request(self, url, method='GET', params=None, data=None, batch=False):

        if self.collect_requests:
            body = params if method in ('GET', 'HEAD') else data
            self._requests.append(
                BatchRequest(id=None, method=method, path=url, body=body)
            )

            self.log.info(
                'Prepared for batch IDM request method=%s, url=%s, body=%s',
                method, url, body
            )
            return

        if data is not None:
            data = json.dumps(data)

        request = self._prepare_request(url, method, params, data)
        response = self._process_request(request, batch)

        if not response.text:
            return {}

        try:
            result = response.json()
        except ValueError:
            self.log.warning('Non-json answer from IDM: %s', response.text)
            raise exceptions.BadRequest(extra=response.text)

        if batch:
            successful = []
            failed = []

            for subresponse in result['responses']:
                # body может приехать пустой
                subresult = subresponse['body'] or {}
                subresult['request'] = self.prepared_requests.get(int(subresponse['id']))
                if subresponse['status_code'] >= 400:
                    failed.append(subresult)
                else:
                    successful.append(subresult)

            result = BatchResult(
                ok=response.ok,
                successful=successful,
                failed=failed,
            )

        return result

    def _prepare_request(self, url, method, params=None, data=None):
        url = urljoin(self.end_point, url)

        if params is None:
            params = {}

        # убираем токен из логов
        self.log.info(
            'IDM request method=%s, url=%s, params=%s, data=%s',
            method, url, params, data
        )

        headers = {
            'Content-type': 'application/json',
            'X-Ya-Service-Ticket': get_tvm_ticket(destination=str(settings.IDM_TVM_ID))
        }

        return requests.Request(
            url=url,
            method=method,
            headers=headers,
            params=params,
            data=data,
        )

    def _process_request(self, request, batch=False, retry=None):
        retry = retry or self.retry
        try:
            response = self._run_request(request)
        except requests.exceptions.Timeout:
            self.log.warning(
                'IDM request timeout at %s with params %s, retry %s',
                request.url, request.params, retry
            )

            raise exceptions.TimeoutError()

        except requests.RequestException as e:
            self.log.info(
                'IDM request error %s at %s with params %s, retry %s',
                e, request.url, request.params, retry
            )

            if retry:
                return self._process_request(request, batch, retry-1)
            else:
                raise exceptions.BadRequest(extra=e)

        if not response.ok:
            if response.headers.get('X-IDM-READONLY') == 'true':
                raise exceptions.ReadonlyError(extra=self.fetch_error(response))

            errors = {
                400: exceptions.BadRequest,
                401: exceptions.AuthError,
                403: exceptions.Forbidden,
                404: exceptions.NotFound,
                409: exceptions.Conflict,
                502: exceptions.UnavailableError,
                503: exceptions.UnavailableError,
                504: exceptions.TimeoutError,
            }
            extra = self.fetch_error(response)

            if response.status_code == 409 and 'сломана' in extra.get('message', ''):
                raise exceptions.SystemBroken(extra=extra)

            if response.status_code in errors:
                error_class = errors.get(response.status_code)
                # на 401 / 403 IDM отвечает пустым body
                kwargs = {
                    'extra': extra,
                }

                if 'message' in extra:
                    kwargs['message'] = extra['message']

                raise error_class(**kwargs)

            elif response.status_code < 500:
                self.log.warning(
                    'IDM returned 400 for request: %s %s',
                    request.method,
                    request.url,
                )
                raise exceptions.BadRequest(extra=extra)

            elif not batch:
                # ручка батчей отдает 500 если хоть один из запросов в батче
                # был 4XX/5XX, но чтоб понять какой из запросов это был
                # нужно закончить обработку батча
                self.log.warning(
                    'IDM returned 500 for request %s %s: %s',
                    request.method,
                    request.url,
                    response.content[:500],
                )
                raise exceptions.IDMError(extra=extra)

        return response

    def _run_request(self, request):
        return self.session.send(request.prepare(), timeout=self.timeout)

    @staticmethod
    def fetch_error(response):
        default_message = 'IDM did not provide an error message'
        params = {
            'message': default_message,
            'raw': response.text,
        }

        if not response.content:
            return params

        try:
            json = response.json()
        except ValueError:
            params['message'] = 'IDM answer is not a json'
            return params

        if isinstance(json.get('error'), str):
            params['message'] = json['error']

        elif isinstance(json.get('error'), dict):
            params['message'] = json['error'].get('message', default_message)

        elif json.get('message'):
            params['message'] = json['message']

        params['errors'] = json.get('errors')

        # hack
        if response.status_code == 409:
            message = params.get('message')
            if message is not None and ALREADY_HAS_ROLE_TOKEN in message:
                end_index = message.index(ALREADY_HAS_ROLE_TOKEN) + len(ALREADY_HAS_ROLE_TOKEN)
                params['message'] = message[:end_index]

        return params

    def head(self, url, **kwargs):
        return self._request(url, 'HEAD', **kwargs)

    def get(self, url, **kwargs):
        return self._request(url, 'GET', **kwargs)

    def delete(self, url, data={}, **kwargs):
        return self._request(url, 'DELETE', data=data, **kwargs)

    def post(self, url, data, **kwargs):
        return self._request(url, 'POST', data=data, **kwargs)

    def patch(self, url, data, **kwargs):
        return self._request(url, 'PATCH', data=data, **kwargs)

    def put(self, url, data, **kwargs):
        return self._request(url, 'PUT', data=data, **kwargs)

    def batch(self, _requests, **kwargs):

        for _id, _request in enumerate(_requests):

            if not isinstance(_request, BatchRequest):
                _request = BatchRequest(id=_id, **_request)
            else:
                _request.id = _id

            self.prepared_requests[_id] = attr.asdict(_request)

        return self.post('batch/', list(self.prepared_requests.values()), batch=True)

    def __enter__(self):
        self.collect_requests = True
        self._requests = []
        self.batch_result = None
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.collect_requests = False
        self.batch_result = self.batch(self._requests)

    def set_is_frozen(self, login: str, value: bool):
        return self.patch(
            f'users/{login}/',
            data={'is_frozen': value},
        )


def idm_manager(timeout: Optional[int] = None, retry: Optional[int] = None):
    manager = Manager(
        end_point=settings.ABC_IDM_ENDPOINT,
        timeout=timeout,
        retry=retry,
    )
    return manager
