from typing import Optional, Dict, Union, Any

from attr import dataclass, attrib

from django.conf import settings
from django.db.models import Q

from plan.roles.models import Role
from plan.services.models import Service, ServiceTag
from plan.idm.adapters import RoleNodeManager
from plan.idm.exceptions import BadRequest, NotFound
from plan.idm.manager import idm_manager


@dataclass(frozen=True)
class Node(object):
    """
    Модель узла дерева ролей в IDM.
    """
    slug: str
    unique_id: str
    is_exclusive: bool
    name: Optional[Union[str, Dict[str, str]]] = attrib(default=None)
    parent: Optional['Node'] = attrib(default=None)
    set: Optional[str] = attrib(default=None)
    system: str = attrib(default=settings.ABC_IDM_SYSTEM_SLUG)
    review_required: Optional[str] = attrib(default=None)

    def __str__(self):
        return self.slug_path

    @property
    def key(self):
        if self.slug != '*':
            return '{}_key'.format(self.slug)
        else:
            return 'role'

    @property
    def ancestry(self):
        node = self
        ancestors = []
        while node.parent:
            ancestors.append(node.parent)
            node = node.parent
        return reversed(ancestors)

    @property
    def slug_path(self):
        ancestry = ['{}/{}'.format(ancestor.slug, ancestor.key) for ancestor in self.ancestry]
        ancestry.append(self.slug)
        return '/type/' + '/'.join(ancestry) + '/'

    @property
    def key_path(self):
        return self.slug_path + self.key + '/'

    @property
    def value_path(self):
        ancestry = [ancestor.slug for ancestor in self.ancestry]
        ancestry.append(self.slug)
        return '/' + '/'.join(ancestry) + '/'

    def register(self, manager=None, raise_on_duplicate=True):
        """
        Создает узел дерева ролей в IDM.
        https://wiki.yandex-team.ru/intranet/idm/api/public/#dobavlenie-postprefix/rolenodes/
        """
        if manager is None:
            manager = idm_manager()

        kwargs = {
            'manager': manager,
            'system': self.system,
        }

        created_slug = created_key = False

        # создаем основной узел
        try:
            RoleNodeManager.create(**dict(
                kwargs,
                parent=self.parent.key_path,
                slug=self.slug,
                name=self.name,
                is_exclusive=self.is_exclusive,
                unique_id=self.unique_id,
                set=self.set,
                review_required=self.review_required
            ))
            created_slug = True
        except BadRequest:
            if raise_on_duplicate:
                raise

        # создаем узел ключа
        # но для конечных нод ролей это делать не нужно
        if self.parent.slug != '*':
            try:
                RoleNodeManager.create(**dict(
                    kwargs,
                    parent=self.slug_path,
                    slug=self.key,
                    name=self.name,
                ))
                created_key = True
            except BadRequest:
                if raise_on_duplicate:
                    raise

        return created_slug, created_key

    def fetch_data(self, manager) -> Dict[str, Any]:
        """
        Получает данные узла IDM
        https://wiki.yandex-team.ru/intranet/idm/api/public/#chtenie1
        """
        if manager is None:
            manager = idm_manager()
        return RoleNodeManager.get(
            manager=manager,
            system=self.system,
            node_path=self.slug_path,
        )

    def delete(self, manager=None):
        """
        Удаляет узел и всё его поддерево.
        https://wiki.yandex-team.ru/intranet/idm/api/public/#udalenie-deleteprefix/rolenodes/path/
        """
        if manager is None:
            manager = idm_manager()

        RoleNodeManager.delete(
            manager=manager,
            system=self.system,
            node_path=self.slug_path,
        )

    def update(self, manager=None, **kwargs):
        """
        Апдейтит ноду в idm.
        https://wiki.yandex-team.ru/intranet/idm/api/public/#redaktirovanie-putprefix/rolenodes/path/
        """
        if manager is None:
            manager = idm_manager()

        RoleNodeManager.update(
            manager=manager,
            system=self.system,
            node_path=self.slug_path,
            **kwargs
        )

    def exists(self, manager=None) -> bool:
        """
        Проверяем, что нода есть в IDM.
        Возвращает, существует ли нода или нет.
        """
        if manager is None:
            manager = idm_manager()

        try:
            RoleNodeManager.get(
                manager=manager,
                system=self.system,
                node_path=self.slug_path,
            )
            # у ролей value-ноду проверять не надо
            if self.parent.slug != '*':
                RoleNodeManager.get(
                    manager=manager,
                    system=self.system,
                    node_path=self.key_path,
                )
            return True

        except NotFound:
            return False


SERVICE_PARENT_NODE = Node(
    parent=None,
    slug='services',
    name=None,
    is_exclusive=False,
    unique_id='services',
    set=None,
)
SERVICE_TAG_PARENT_NODE = Node(
    system=settings.ABC_EXT_IDM_SYSTEM_SLUG,
    parent=None,
    slug='service_tags',
    name=None,
    is_exclusive=False,
    unique_id='service_tags',
    set=None,
)


def build_service_node(service: Service, parent: Node) -> Node:
    return Node(
        parent=parent,
        slug=service.slug,
        name={'ru': service.name, 'en': service.name_en},
        is_exclusive=False,
        unique_id='service_{}'.format(service.pk),
        set=None,
    )


def get_service_node(service):
    # ключевой узел сервиса
    node = SERVICE_PARENT_NODE

    if service:
        for family_member in service.get_ancestors(include_self=True):
            node = build_service_node(family_member, node)

    return node


def get_service_roles_node(service: Service) -> Node:
    # зонтичная нода для всех ролей в сервисе
    return Node(
        parent=get_service_node(service),
        slug='*',
        name='Роли сервиса',
        is_exclusive=False,
        unique_id='service_{}_roles'.format(service.pk),
        set=None,
    )


def get_role_node(service: Service, role: Role) -> Node:
    # нода конкретной роли в сервисе
    return Node(
        parent=get_service_roles_node(service),
        slug=str(role.id),
        name={'ru': role.name, 'en': role.name_en},
        is_exclusive=(role.code == Role.EXCLUSIVE_OWNER),
        unique_id='service_{}_role_{}'.format(service.id, role.id),
        set=str(role.id),
        review_required=service.review_required,
    )


def get_service_role_nodes(service):
    # ноды каждой роли, которая есть в сервисе
    roles = (
        Role.objects
        .filter(Q(service=None) | Q(service=service))
        .order_by('id').only('id', 'name', 'name_en')
    )

    for role in roles:
        yield get_role_node(service, role)


def find_missing_nodes(nodes, manager=None):
    """
    Находит ноды ролей, которых нет в idm.
    """
    if manager is None:
        manager = idm_manager()

    for node in nodes:
        if not node.exists(manager=manager):
            yield node


def get_service_from_value_path(value_path):
    from plan.services.models import Service
    for value in reversed(value_path.split('/')):
        if not value or value.isdigit() or value == '*' or value == 'services':
            continue
        else:
            return Service.objects.get(slug=value)

    raise ValueError("Couldn't parse a service slug from the value path!")


def remove_service_tag_node(service_tag: ServiceTag):
    service_tag_node = Node(
        system=settings.ABC_EXT_IDM_SYSTEM_SLUG,
        parent=SERVICE_TAG_PARENT_NODE,
        slug=service_tag.slug,
        name={'ru': service_tag.name, 'en': service_tag.name_en},
        is_exclusive=False,
        unique_id='service_tag_{}'.format(service_tag.pk),
        set=None,
    )
    service_tag_node.delete()
