from dateutil import parser

import attr

from django.conf import settings

from plan.idm import nodes
from plan.idm.adapters import RoleManager
from plan.idm.manager import idm_manager
from plan.roles import models as role_models
from plan.staff.models import Department, Staff


@attr.s(frozen=True)
class Role(object):
    role = attr.ib()
    service = attr.ib()

    idm_id = attr.ib()
    state = attr.ib()

    granted_at = attr.ib()

    person = attr.ib(default=None)
    department = attr.ib(default=None)

    @classmethod
    def from_idm(cls, idm_data, service=None, users=None, departments=None, roles=None):
        granted_at = idm_data.get('granted_at')

        user = None
        department = None
        service = service or nodes.get_service_from_value_path(idm_data['node']['value_path'])

        if idm_data.get('user'):
            if users:
                user = users[idm_data['user']['username']]
            else:
                user = Staff.objects.get(login=idm_data['user']['username'])

        elif idm_data.get('group'):
            if departments:
                department = departments[idm_data['group']['id']]
            else:
                department = Department.objects.get(staff_id=idm_data['group']['id'])

        if roles:
            role = roles[int(idm_data['node']['slug'])]
        else:
            role = role_models.Role.objects.get(id=int(idm_data['node']['slug']))

        return cls(
            person=user,
            department=department,
            role=role,
            service=service,
            state=idm_data['state'],
            idm_id=idm_data['id'],
            granted_at=None if granted_at is None else parser.parse(granted_at)
        )


def get_requested_roles_for_service(service, manager=None):
    if manager is None:
        manager = idm_manager()

    roles_node = nodes.get_service_roles_node(service)

    roles_json = RoleManager.get_roles(
        manager=manager,
        filters={
            'system': settings.ABC_IDM_SYSTEM_SLUG,
            'path': roles_node.value_path,
            'type': 'requested'
        }
    )
    roles_list = roles_json['objects']
    data_for_roles_list = prepare_data_for_roles_list(roles_list)
    data_for_roles_list['service'] = service
    for role_data in roles_json['objects']:
        yield Role.from_idm(role_data, **data_for_roles_list)


def get_service_group_roles_count(service, manager=None):
    if manager is None:
        manager = idm_manager()

    group_ids = list(
        service
        .get_descendants(include_self=True)
        .values_list('staff_id', flat=True)
    )

    roles_json = RoleManager.get_roles(
        manager=manager,
        filters={
            'group': ','.join(str(group_id) for group_id in group_ids),
            'type': 'active',
            'limit': 1
        },
        no_meta=False,
    )

    return roles_json['total_count']


def get_granted_roles(service, role, manager=None):
    if manager is None:
        manager = idm_manager()

    role_node = nodes.get_role_node(service, role)

    roles_json = RoleManager.get_roles(
        manager=manager,
        filters={
            'system': settings.ABC_IDM_SYSTEM_SLUG,
            'type': 'active',
            'path': role_node.value_path
        }
    )
    roles_list = roles_json['objects']
    data_for_roles_list = prepare_data_for_roles_list(roles_list)
    data_for_roles_list['service'] = service
    for role_data in roles_json['objects']:
        yield Role.from_idm(role_data, **data_for_roles_list)


def prepare_data_for_roles_list(roles_list):
    users = set()
    departments = set()
    roles = set()
    for role_data in roles_list:
        if role_data.get('user'):
            users.add(role_data['user']['username'])
        elif role_data.get('group'):
            departments.add(role_data['group']['id'])

        roles.add(int(role_data['node']['slug']))

    users = {x.username: x for x in Staff.objects.filter(login__in=users)}
    departments = {x.staff_id: x for x in Department.objects.filter(staff_id__in=departments)}
    roles = role_models.Role.objects.in_bulk(roles)

    return {'users': users, 'departments': departments, 'roles': roles}
