import logging

from django.conf import settings
from django.contrib.auth.models import Permission

log = logging.getLogger(__name__)


def _get_perm(perm_codename):
    if '.' in perm_codename:
        app_label, model, code = perm_codename.split('.')
        perm = Permission.objects.get(
            codename=code,
            content_type__app_label=app_label,
            content_type__model=model)

    else:
        perm = Permission.objects.get(
            codename=perm_codename,
            content_type__app_label='internal_roles')

    return perm


def add_perm(perm_codename, staff):
    log.info('Add perm %s to %s', perm_codename, staff.user.username)

    perm = _get_perm(perm_codename)
    staff.user.user_permissions.add(perm)


def assign_perms_for_internal_role(role_name, staff):
    permissions = settings.ABC_INTERNAL_ROLES_PERMISSIONS.get(role_name, [])

    for perm_codename in permissions:
        add_perm(perm_codename, staff)


def remove_perm(perm_codename, staff):
    log.info('Remove perm %s from %s', perm_codename, staff.user.username)

    perm = _get_perm(perm_codename)
    staff.user.user_permissions.remove(perm)


def remove_perms_for_internal_role(role_name, staff):
    role_permissions = set(settings.ABC_INTERNAL_ROLES_PERMISSIONS.get(role_name, []))
    existing_roles = staff.internalrole_set.values_list('role', flat=True)
    remaining_permissions = set()
    for role in existing_roles:
        remaining_permissions.update(settings.ABC_INTERNAL_ROLES_PERMISSIONS.get(role, []))

    for perm_codename in (role_permissions - remaining_permissions):
        remove_perm(perm_codename, staff)
