import json
import logging

import yenv
from django.http import Http404, HttpResponse, HttpResponseRedirect
from django.views.generic import View

from plan.api.exceptions import ABCAPIException, handler
from plan.common.utils.auth import check_authentication
from plan.common.utils.collection.mapping import filter_dict
from plan.common.utils.strings import camelcase_to_underscore
from plan.exceptions import HttpRedirect, Error

logger = logging.getLogger(__name__)


class UniversalView(View):
    """
    Супермегабазовая вьюха для всего проекта.

    Особенности:
      * render_to — легкое переключение формата, в который нужно рендерить
        результат (сейчас в json или privjs, но можно добавить рендеринг в
        какой-нибудь django-шаблон для отладки или служебных страниц).
        Это позволяет плавно мигрировать, если прикрутить к privjs-ручке эту
        вьюху в качестве базовой, то можно впоследствии просто изменить
        render_to. (скорее всего все не так идеально, так как эксепшны
        обрабатываются по-разному, но это тоже можно дописать.)
      * dispatch переопределен, так чтобы из методов get, post, etc. нужно
        возвращать только dict с данными, не вызывая render-куда-то-там.
        (я может быть что-то упускаю, но пока не придумал)
      * стандартный формат ручек: то есть все что вернул get/post кладется в
        content, все что срайзило ошибку Error кладется в error. Остальные
        исключения также перехватываются и заворачиваются в error.
    """

    render_to = 'json'  # 'json'|'priv'| 'html'
    RENDER_TYPE_MAPPING = {}
    # временная штука, изначально предполагалась только для privjs
    page_name = 'DUMMY'
    template_name = None  # Для джанго-шаблонов (render_to = 'html')

    def dispatch(self, request, *args, **kwargs):
        """
        Переопределяем dispatch, чтобы из методов get/post и прочих можно было
        бы возвращать просто dict вместо HttpResponse.
        TODO: Понять всегда ли это удобно?
        """
        base_dispatch = super(UniversalView, self).dispatch
        try:
            check_authentication(request)
            method_results = base_dispatch(request, *args, **kwargs)

        except Error as exc:
            return self.render_standard_response(error=exc)

        # Не обрабатываем Http404, потому что его обработает django
        except Http404:
            raise

        except HttpRedirect as exc:
            return HttpResponseRedirect(exc.redirect_to)

        except ABCAPIException as exc:
            return handler.abc_exception_handler(exc, None)

        except Exception as exc:
            error_code = camelcase_to_underscore(exc.__class__.__name__)
            logger.exception('Unhandled exception in view.')
            return self.render_standard_response(error=Error(
                error_code=error_code.upper(),
                message=getattr(exc, 'message', str(exc)),
            ))

        if isinstance(method_results, HttpResponse):
            return method_results

        return self.render_standard_response(content=method_results)

    def render_standard_response(self, content=None, error=None):
        if content and error:
            raise Exception("`content` and `error` can't be given together")

        if content is None and error is None:
            raise Exception("`content` or `error` should be given")

        # супер гайдлайн формат.
        if error:
            status = error.http_code
            context = {
                'content': {},
                'error': {
                    'code': error.error_code,
                    'message': getattr(error, 'message', str(error)),
                    'params': error.params,
                },
            }
        else:
            status = 200
            context = {
                'content': content,
                'error': {},
            }

        return self.render_to_response(context=context, status=status)

    def render_to_response(self, context, status):
        rendering_type = self.detect_rendering_type()

        renderer_cls = {
            'html': HTMLRenderer,
            'json': JSONRenderer,
            'priv': PrivRenderer,
        }.get(rendering_type)

        if rendering_type == 'html':
            renderer = renderer_cls(
                request=self.request,
                template_name=self.template_name,
            )
        elif rendering_type in ('json', 'priv'):
            renderer = renderer_cls(
                request=self.request,
                page_name=self.page_name,
            )
        else:
            raise Exception('Unknown rendering type `%s`' % rendering_type)

        return renderer.render(
            render_params=self.render_params,
            context=context,
            status=status,
        )

    def detect_rendering_type(self):
        if yenv.type == 'development' and '_render.to' in self.request.GET:
            return self.request.GET['_render.to']
        else:
            return self.RENDER_TYPE_MAPPING.get(
                self.request.method.lower(),
                self.render_to
            )

    @property
    def render_params(self):
        all_params = self.request.GET
        return filter_dict(
            mapping=all_params,
            keys=[k for k in all_params.keys() if k.startswith('_render.')]
        )

    @property
    def json_data(self):
        try:
            return json.load(self.request)
        except ValueError:
            raise Error(
                error_code='JSON_DECODE_ERROR',
                message='No JSON object could be decoded',
            )


class Renderer(object):

    def __init__(self, request, *args, **kwargs):
        self.request = request

    def render(self, context, render_params, **kwargs):
        raise NotImplementedError


class HTMLRenderer(Renderer):

    def __init__(self, request, template_name):
        super(HTMLRenderer, self).__init__(request=request)
        self.template_name = template_name

    def render(self, context, render_params, **kwargs):
        from django.shortcuts import render

        return render(
            request=self.request,
            template_name=self.template_name,
            dictionary=context,
            **kwargs
        )


class JSONRenderer(Renderer):

    # NOTE: @rodique сказал, что page_name в json нужен временно.
    # Не забыть убрать, когда можно будет.
    def __init__(self, request, page_name):
        super(JSONRenderer, self).__init__(request=request)
        self.page_name = page_name

    def render(self, context, render_params, **kwargs):
        # пока переиспользуем код из миксина, чтобы не дублировать.
        # когда все вьюхи будут наследованы от UniversalView, его нужно
        # влить в рендерер. Или можно наоборот -- в миксине заюзать Рендерер.
        from plan.lib.views.mixins import JSONResponseMixin as Renderer

        return Renderer().render_to_response(
            context=context,
            **kwargs
        )


class PrivRenderer(Renderer):

    def __init__(self, request, page_name):
        super(PrivRenderer, self).__init__(request=request)
        self.page_name = page_name

    def render(self, context, render_params, **kwargs):
        from django.shortcuts import render

        context.update({'page_type': self.page_name})
        return render(
            request=self.request,
            template_name='desktop.bundles',
            dictionary=context,
            **kwargs
        )
