import datetime
import decimal
import json
import logging

from django import http
from django.utils.functional import Promise as DjangoPromise

logger = logging.getLogger(__name__)


class JSONResponseMixin(object):
    def render_to_response(self, context, status=200):
        """Returns a JSON response containing 'context' as payload"""
        return self.get_json_response(
            content=self.convert_context_to_json(context),
            status=status
        )

    def get_json_response(self, content, **httpresponse_kwargs):
        """Construct an `HttpResponse` object."""
        return http.HttpResponse(
            content=content,
            content_type='application/json; charset=utf-8',
            **httpresponse_kwargs
        )

    def convert_context_to_json(self, context):
        """Convert the context dictionary into a JSON object"""
        return json.dumps(
            obj=context,
            default=self.serialize_unserializable,
            ensure_ascii=False,
        )

    @staticmethod
    def serialize_unserializable(obj):
        """
        Метод, который подготавливает несериализуемые данные типа дат в
        сериализуемые типа строк.
        """
        if isinstance(obj, datetime.date):
            return obj.isoformat()
        elif isinstance(obj, datetime.datetime):
            from plan.common.utils.dates import datetime_isoformat
            return datetime_isoformat(obj)
        elif isinstance(obj, decimal.Decimal):
            return float(obj)
        elif isinstance(obj, DjangoPromise):
            return str(obj)
        elif isinstance(obj, map):
            return list(obj)
        raise TypeError(obj)


class StdCtxMixin(object):
    def context(self, content=None, errors=None, messages=None):
        return {
            'content': content,
            'errors': errors or [],
            'messages': messages or [],
        }

    def render_std_response(self, content=None, errors=None, messages=None,
                            status=200):
        # TODO: если errors is not None логично ставить статус 409.
        context = self.context(content, errors, messages)

        from plan.common.utils.readonly import add_readonly_context
        add_readonly_context(self.request, context)

        return self.render_to_response(context, status=status)
