from collections import OrderedDict, defaultdict
from datetime import timedelta
import logging

import bson
import celery
import datetime
import dateutil
import waffle

from django.apps import apps
from django.conf import settings
from django.core import exceptions
from django.db import transaction
from django.db.models import Count, Exists, OuterRef, Q
from django.utils import timezone

from closuretree.models import ClosureModel

from plan.api.idm import actions
from plan.celery_app import app
from plan.common.person import Person
from plan.common.utils import oauth, tasks
from plan.common.utils.http import Session
from plan.common.utils.tasks import lock_task
from plan.idm import (
    exceptions as idm_exceptions,
    nodes as idm_nodes,
    roles as idm_roles
)
from plan.idm.manager import idm_manager
from plan.notify.shortcuts import send_to_team, deliver_email
from plan.services import (
    models as services_models,
    tasks as services_tasks
)
from plan.roles import models as roles_models
from plan.roles.tasks import deprive_role
from plan.staff.constants import AFFILIATION, GENDER
from plan.staff.models import Department, Staff
from plan.oebs.models import OEBSAgreement
from plan.oebs.constants import STATES
from plan.oebs.utils import oebs_request_failed

log = logging.getLogger(__name__)
TEST_STAFF_API_URL = 'https://staff-api.test.yandex-team.ru/v3/persons'


@tasks.lock_task
def test_person_copy(login, cookies):
    """Функция копирует человека из тестового стаффа
    в планнерную реплику продакшенового стаффа по его логину.
    Обращение в тестовый стафф происходит через апи.
    Функция имеет значение для любого непродакшенового окружения.
    """
    url = TEST_STAFF_API_URL
    params = {
        'login': login,
        '_one': 1,
    }
    # нет смысла делать обработку ошибок. поскольку результат
    # никуда не передается, в случае необходимости пользователь/
    # тестировщик повторит свой запрос

    with Session() as session:
        result = session.get(url, params=params, cookies=cookies)

    person_data = result.json()
    try:
        department_id = person_data['department_group']['department']['id']
        department_obj = Department.objects.get(id=department_id)
    except (KeyError, exceptions.ObjectDoesNotExist):
        department_obj = Department.objects.first()
    Staff.objects.create(
        login=login, normal_login=login,
        login_ld=login, login_mail=login,
        first_name=person_data['name']['first']['ru'],
        last_name=person_data['name']['last']['ru'],
        affiliation=AFFILIATION.YANDEX,
        gender=GENDER.MALE,
        department=department_obj,
        join_at=dateutil.parser.parse(person_data['official']['join_at']),
        created_at=dateutil.parser.parse(person_data['created_at']),
        modified_at=timezone.now(),
    )


@app.task
@transaction.atomic
def rebuild_model_tree(model):
    cls = apps.get_model(model)
    if issubclass(cls, ClosureModel):
        cls.rebuildtable()


@tasks.lock_task
def cleanup_pidbox_replies():
    """
    Если клиент, ожидающий ответов на своё широковещательное сообщение,
    умрёт не своей смертью, в messages.routing навсегда останется
    временный маршрут к очереди для ответов, а в messages останутся ответы.

    Эта задача удаляет временные маршруты и ответы, созданные час назад и
    ранее.
    """
    with celery.current_app.default_connection() as connection:
        db = connection.channel().client

        treshold_time = timezone.now() - timedelta(hours=1)
        treshold_oid = bson.ObjectId.from_datetime(treshold_time)

        # delete old broadcast reply routes
        db['messages.routing'].remove({
            'exchange': 'reply.celery.pidbox',
            '_id': {
                '$lt': treshold_oid,
            },
        })

        # delete old broadcast replies
        db['messages'].remove({
            'queue': {
                '$regex': r'\.reply\.celery\.pidbox$',
            },
            '_id': {
                '$lt': treshold_oid,
            },
        })


def missing_nodes_after_register(missing_main_nodes, manager):
    """
    Если каких-то нод нет, попробуем добавить.
    Возвращаем ноды, которых нет даже после досоздания.
    """

    problem_node = []
    for node in missing_main_nodes:
        try:
            logging.info('Node {} is missing from IDM.'.format(node.slug_path))
            node.register(manager)

        except idm_exceptions.IDMError:
            logging.info('It is impossible to create a node {}.'.format(node.slug_path))
            problem_node.append(node)

    if len(missing_main_nodes) != len(problem_node):
        # проверим после досоздания
        missing_main_nodes = list(idm_nodes.find_missing_nodes(missing_main_nodes, manager=manager))

    return missing_main_nodes


@lock_task
def check_nodes(verbose=0):
    log.info('Check nodes against IDM')
    manager = idm_manager()

    stats = OrderedDict()
    for service in services_models.Service.objects.active().order_by('level'):
        main_nodes = (
            idm_nodes.get_service_node(service),
            idm_nodes.get_service_roles_node(service),
        )

        role_nodes = idm_nodes.get_service_role_nodes(service)

        # Если нет родительской ноды, то и этой не будет
        if service.parent_id and service.parent_id in stats:
            stats[service.id] = {
                'service': service,
                'missing_nodes': main_nodes,
                'missing_role_nodes': role_nodes
            }
            continue

        final_missing_main_nodes = []
        final_missing_role_nodes = []
        # Если сервисной ноды нет, то нет смысла проверять ролевые
        missing_main_nodes = list(idm_nodes.find_missing_nodes(main_nodes, manager=manager))
        if missing_main_nodes:
            # попробуем добавить
            final_missing_main_nodes = missing_nodes_after_register(missing_main_nodes, manager)
            if final_missing_main_nodes:
                final_missing_role_nodes = role_nodes

        if not final_missing_role_nodes:
            missing_role_nodes = list(idm_nodes.find_missing_nodes(role_nodes, manager=manager))
            if missing_role_nodes:
                final_missing_role_nodes = missing_nodes_after_register(missing_role_nodes, manager)

        if final_missing_main_nodes or final_missing_role_nodes:
            stats[service.id] = {
                'service': service,
                'missing_nodes': final_missing_main_nodes,
                'missing_role_nodes': final_missing_role_nodes
            }

    if stats:
        send_to_team(
            'notifications.maintenance.check_nodes',
            context={'stats': stats}
        )


def find_requests(request_model, service_ids, check_agreement=False):
    result = request_model.objects.active().filter(
        updated_at__lte=timezone.now() - datetime.timedelta(minutes=15),
    )
    if check_agreement:
        result = result.filter(
            Q(oebsagreement__isnull=True) | Q(oebsagreement__state__in=STATES.FINAL_STATES)
        )
    result = set(result.values_list('service_id', flat=True))

    if service_ids:
        result = result & set(service_ids)

    return result


@lock_task
def clear_readonly(service_ids=None):
    """
    Таска допинывалка.
    Тут может быть три типа сервиса:
        * которые зависли с readonly_state
        * у которых нет статуса, но есть:
            - активные ServiceCreateRequest
            - активные ServiceMoveRequest
    """
    queryset = services_models.Service.objects.filter(
        readonly_state__isnull=False,
        modified_at__lte=timezone.now()-datetime.timedelta(minutes=15),
    ).annotate(
        oebs_agreement_exists=Exists(
            OEBSAgreement.objects.active().filter(
                service_id=OuterRef('id'),
            )
        )
    ).filter(oebs_agreement_exists=False)

    if service_ids:
        queryset = queryset.filter(id__in=service_ids)

    services = {state[0]: set() for state in services_models.Service.READONLY_STATES}
    for service in queryset.only('id', 'readonly_state'):
        services[service.readonly_state].add(service.id)

    # добавляем в дикт сервисы без readonly статуса, но с зависшими ServiceCreateRequest
    created = find_requests(services_models.ServiceCreateRequest, service_ids)
    services[services_models.Service.CREATING] |= created

    # добавляем в дикт сервисы без readonly статуса, но с зависшими ServiceMoveRequest
    moved = find_requests(services_models.ServiceMoveRequest, service_ids, check_agreement=True)
    services[services_models.Service.MOVING] |= moved

    # если завис в переименовании - значит переименование в idm не удалось
    # и нужно по логам посмотреть в чем дело, в abc readonly_state снимаем
    renaming_services = services[services_models.Service.RENAMING]
    if renaming_services:
        services_models.Service.objects.filter(
            pk__in=renaming_services
        ).update(readonly_state=None)
        log.error('Changing readonly_state(None) on %s - renaming in idm failed', renaming_services)

    for service_id in services[services_models.Service.CREATING]:
        finish_creating.delay(service_id)

    for service_id in services[services_models.Service.MOVING]:
        finish_moving.delay(service_id)

    for service_id in services[services_models.Service.DELETING]:
        finish_deleting.delay(service_id)

    for service_id in services[services_models.Service.CLOSING]:
        finish_closing.delay(service_id)


@lock_task
def check_double_roles():
    queryset = (
        services_models.ServiceMember.objects.filter(from_department=None)
        .values('staff_id', 'service_id', 'role_id')
        .annotate(amount=Count(['staff_id', 'service_id', 'role_id']))
        .filter(amount__gt=1)
    )

    doubles = []
    for record in queryset:
        doubles.append(list(
            services_models.ServiceMember.objects.filter(
                service_id=record['service_id'],
                staff_id=record['staff_id'],
                role_id=record['role_id'],
                from_department=None,
            )
        ))

    if doubles:
        send_to_team('notifications.services.management.check_roles', {'doubles': doubles})


@app.task(bind=True)
@transaction.atomic
def finish_creating(self, service_id):
    service = services_models.Service.objects.get(id=service_id)
    qs = services_models.ServiceCreateRequest.objects.filter(service=service)
    log.info("finish creating service: %s", service.slug)

    if qs.active().exists():
        request = qs.active().get()
        initial_processing_states = (
            services_models.ServiceCreateRequest.REQUESTED,
            services_models.ServiceCreateRequest.APPROVED,
            services_models.ServiceCreateRequest.PROCESSING_IDM
        )
        if request.state in initial_processing_states:
            services_tasks.register_service.delay(request.id)

        elif request.state == services_models.ServiceCreateRequest.PROCESSING_HEAD:
            services_tasks.request_service_head.delay(request.id)

        elif request.state == services_models.ServiceCreateRequest.PROCESSING_ABC:
            services_tasks.finalize_service_creation.delay(request.id)

        else:
            # переведем в PROCESSING_IDM, тогда в register_service будет проверено дерево
            # если узлы уже есть, то add_service не будет их создавать, просто пойдём дальше
            request.process_idm()
            request.save()
            services_tasks.register_service.delay(request.id)

    elif qs.exists():
        fix_service_idm_nodes.delay(service_id, drop_readonly=True)

    else:
        request = services_models.ServiceCreateRequest.request(
            service=service,
            requester=Person(oauth.get_abc_zombik()),
        )
        services_tasks.register_service.apply_async(
            args=(request.id,),
            countdown=settings.ABC_DEFAULT_COUNTDOWN
        )


@app.task(bind=True)
@transaction.atomic
def finish_moving(self, service_id):
    service = services_models.Service.objects.get(id=service_id)
    if service.oebs_agreements.active().exists():
        log.info(f'Active OEBSAgreement exists, skipping finish_moving: {service.slug}')
        return

    log.info("finish moving service: %s", service.slug)
    approved_moves = service.move_requests.approved().active()

    if approved_moves:
        move_request = approved_moves.latest()

        if oebs_request_failed(move_request):
            return

        # подвисли в idm
        if move_request.state in (services_models.ServiceMoveRequest.APPROVED,
                                  services_models.ServiceMoveRequest.PROCESSING_IDM,
                                  services_models.ServiceMoveRequest.PROCESSING_OEBS):
            if move_request.start_moving_idm:
                services_tasks.verify_and_move_service.apply_async(args=[move_request.id])
            else:
                services_tasks.move_service.apply_async(args=[move_request.id])

        # подвисли у себя
        elif move_request.state == services_models.ServiceMoveRequest.PROCESSING_ABC:
            if waffle.switch_is_active('async_move_service_abc_side'):
                services_tasks.move_service_abc_side.apply_async(args=[move_request.id])

        # непонятно, что делать, поэтому на всякий случай чекнем и починим дерево
        elif move_request.state == services_models.ServiceMoveRequest.COMPLETED:
            fix_service_idm_nodes(service_id)

    else:
        # По какой-то причине перемещение закончилось, но readonly = moving остался
        # Убираем readonly
        service.readonly_state = None
        service.save(update_fields=['readonly_state'])


@app.task(bind=True)
@transaction.atomic
def finish_deleting(self, service_id):
    service = services_models.Service.objects.get(id=service_id)
    if service.oebs_agreements.active().exists():
        log.info(f'Active OEBSAgreement exists, skipping finish_deleting: {service.slug}')
        return

    service_delete_request = (
        services_models.ServiceDeleteRequest.objects
        .filter(service=service)
        .exclude(state=services_models.ServiceDeleteRequest.COMPLETED)
        .order_by('created_at')
        .last()
    )
    if service_delete_request:
        if oebs_request_failed(service_delete_request):
            return

        node = idm_nodes.get_service_node(service)

        if node.exists():
            services_tasks.delete_service.delay(service_delete_request.id)

        elif not service.is_deleted:
            services_tasks.delete_service_abc_side.delay(service_delete_request.id)


@app.task(bind=True)
@transaction.atomic
def finish_closing(self, service_id):
    service = services_models.Service.objects.get(id=service_id)
    if service.oebs_agreements.active().exists():
        log.info(f'Active OEBSAgreement exists, skipping finish_closing: {service.slug}')
        return

    if service.state != service.states.CLOSED:
        service_close_request = (
            services_models.ServiceCloseRequest.objects
                .filter(service=service)
                .exclude(state=services_models.ServiceCloseRequest.COMPLETED)
                .order_by('created_at')
                .last()
        )
        if service_close_request and not oebs_request_failed(service_close_request):
            services_tasks.close_service.delay(service_close_request.id)


@app.task
@transaction.atomic
def fix_service_idm_nodes(service_id, drop_readonly=False):
    # Чиним узлы дерева сервиса в idm.

    service = services_models.Service.objects.get(id=service_id)

    nodes = (
        idm_nodes.get_service_node(service),
        idm_nodes.get_service_roles_node(service),
    )
    log.info("fix_service_idm_nodes service:%s", service.slug)

    manager = idm_manager()
    missing_nodes = idm_nodes.find_missing_nodes(nodes, manager=manager)
    for node in missing_nodes:
        log.info('Node %s is missing from IDM.', node.slug_path)
        node.register(manager=manager, raise_on_duplicate=False)

    role_nodes = idm_nodes.get_service_role_nodes(service)
    missing_role_nodes = idm_nodes.find_missing_nodes(role_nodes, manager=manager)
    for node in missing_role_nodes:
        log.info('RoleNode %s is missing from IDM.', node.slug_path)
        node.register(manager=manager, raise_on_duplicate=False)

    try:
        actions.add_service_head(service, service.owner, requester=oauth.get_abc_zombik())
    except idm_exceptions.Conflict:
        pass

    if drop_readonly:
        is_in_meta = (
            service.parent is not None and
            service.get_ancestors(include_self=True)
            .filter(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG).exists()
        )

        service.get_descendants(include_self=True).update(
            readonly_state=None,
            readonly_start_time=None,
            is_exportable=not is_in_meta
        )
        service.move_requests.reject()


@app.task
@transaction.atomic
def fix_duplicate_heads():
    owner_role = roles_models.Role.objects.globalwide().get(code=roles_models.Role.EXCLUSIVE_OWNER)

    ids_with_duplicate_heads = (
        services_models.ServiceMember.objects
        .filter(role=owner_role)
        .values('service_id')
        .order_by('service_id')
        .annotate(owner_count=Count('service_id'))
        .filter(owner_count__gt=1)
        .distinct()
    )

    services_with_duplicate_heads = services_models.Service.objects.filter(
        id__in=[data['service_id'] for data in ids_with_duplicate_heads]
    )

    for service in services_with_duplicate_heads:
        heads_in_idm = list(idm_roles.get_granted_roles(service, owner_role))
        heads_in_abc = service.members.heads().order_by('created_at')

        # находим самого нового рукля сервиса в IDM
        most_recent_in_idm = sorted(heads_in_idm, key=lambda role: role.granted_at)[-1]
        logins_of_heads_in_idm = list(role.person.login for role in heads_in_idm)

        duplicate_roles_for_same_person = []
        for head in heads_in_abc:
            # если у человека не нашлось ни одной роли в idm,
            # то удаляем ServiceMember на нашей стороне
            if head.staff.login not in logins_of_heads_in_idm:
                head.deprive()

            # если человек в роли не самый новый руководитель, отзываем роль
            elif head.staff != most_recent_in_idm.person:
                deprive_role.delay(
                    head.id,
                    'Роль отозвана, так как у сервиса было найдено несколько руководителей.'
                )

            else:
                duplicate_roles_for_same_person.append(head)

        # чистим возможные дубликаты роли у самого нового руководителя
        if len(duplicate_roles_for_same_person) > 1:
            duplicate_roles_for_same_person.sort(key=lambda m: m.created_at)
            # самую последнюю роль оставляем
            duplicate_roles_for_same_person.pop()
            # остальные отзываем
            for head in duplicate_roles_for_same_person:
                head.deprive()


@lock_task
def remind_meta_other_heads():
    meta_other = services_models.Service.objects.get(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)

    heads_for_services = defaultdict(set)
    services_in_meta_other = (
        services_models.Service
        .objects
        .active()
        .filter(pk__in=meta_other.get_descendants(), is_exportable=True)
        .values_list('id', flat=True)
    )

    heads = services_models.ServiceMember.objects.heads()

    for head in heads.filter(service_id__in=services_in_meta_other).select_related('service'):
        heads_for_services[head.staff].add(head.service)

    for head, services in heads_for_services.items():
        deliver_email(
            notification_id='notifications.maintenance.remind_meta_other_heads',
            context={"services": services},
            recipients=[head]
        )
