import logging
from datetime import datetime
from typing import Iterable, Union, List

from django.conf import settings
from django.core.serializers.json import DjangoJSONEncoder
from django.db.models import QuerySet
from django.utils import timezone
from django.utils.dateparse import parse_datetime
from errorboosterclient.logbroker import LogbrokerClient, LogbrokerException

from plan.services.models import Service

log = logging.getLogger(__name__)


def serialize_service(service: dict) -> dict:
    """Представляет указанные данные сервиса в виде словаря."""
    service_id = service['id']

    result = {
        'node': 'abc',
        'reference': 'abc_services',
        'id': str(service_id),
        'action': 'put',
        'attrs': {
            'slug': service['slug'],
            'name': {
                'ru': service['name'],
                'en': service['name_en'],
            },
            'state': service['state'],
            'has_children': 'y' if service['children_count'] else 'n',
            'id': service_id,
            'parent_id': service['parent_id'],
        }
    }
    return result


class ExportParams:
    """Интерфейс для параметров экспорта."""

    def __init__(self, since: datetime = None, slugs: List[str] = None):

        if not since:
            since = datetime(1970, 1, 1).replace(tzinfo=timezone.utc)

        self.since = since
        """Дата, по которой следует фильтровать сервисы."""

        self.slugs = slugs or []
        """Псевдонимы сервисов."""

    @classmethod
    def fromdict(cls, params: dict = None) -> 'ExportParams':
        """Альтернативный конструктор для создания объекта параметров экспорта из словаря.

        :param params:

        """
        params = params or {}

        since = params.get('since')

        export_params = ExportParams(
            since=parse_datetime(since) if since else None,
            slugs=params.get('slugs'),
        )

        return export_params

    def todict(self, since: datetime = None) -> dict:
        """Собирает параметры в словарь.

        :param since:

        """
        if not since:
            since = self.since
        return {'since': since.isoformat()}


def pick_services(params: ExportParams) -> Iterable[Union[QuerySet, Service]]:
    """Отбирает сервисы для экспорта, основываясь на переданных параметрах.

    :param params:

    """
    slugs = params.slugs

    if slugs:
        # Указаны псевдонимы конкретных сервисов
        services = Service.objects.filter(slug__in=slugs)

    else:
        # Основная ветвь: отбор по времени изменения.
        services = Service.objects.exportable().alive().filter(
            children_count=0,
            modified_at__gt=params.since,
        )

    services = services.values(
        'id',
        'slug',
        'name',
        'name_en',
        'state',
        'children_count',
        'parent_id',
        'modified_at',

    ).order_by('modified_at')

    return services


def write_to_logbroker(services: Iterable[Service]) -> datetime:
    """Пишет в Logbroker данные об указанных сервисах для MDH.

    Возвращает объект даты-времени самой свежей (по изменению) из ныне
        отправленных записей.

    """
    latest_update_dt = datetime(1970, 1, 1).replace(tzinfo=timezone.utc)

    try:

        with LogbrokerClient(token=settings.MDH_LB_TOKEN) as logbroker:

            producer = logbroker.get_producer(
                source=settings.MDH_LB_SOURCE,
                topic=settings.MDH_LB_TOPIC,
                json_encoder=DjangoJSONEncoder,
            )
            for service in services:
                producer.write(serialize_service(service))

                update_dt = service['modified_at']

                if update_dt > latest_update_dt:
                    latest_update_dt = update_dt

    except LogbrokerException:
        log.error('MDH export malfunction spotted')

    return latest_update_dt
