import pytz
import yenv
from yql.api.v1.client import YqlClient

from django.conf import settings
from django.utils import timezone

from plan.common.utils.tasks import lock_task


YQL_QUERY = r"""
USE hahn;

$patch_match = Re2::Match("^/api(/v[3,4](/[\\w,\\-]+/|/)|frontend(/[\\w,\\-]+/|/)|/)[\\w,\\-]+/[\\w,\\-]+(/\\?.*|/)$");
$capture = Pire::Capture("(/api(/v[3,4](/[\\w,\\-]+/|/)|frontend(/[\\w,\\-]+/|/)|/)[\\w,\\-,\\d]+/[a-z]+/).*");
$details_path_match = Re2::Match("^/api/(v[3,4](/[\\w,\\-]+/|/)|frontend(/[\\w,\\-]+/|/)|/)[\\w,\\-]+/\\d+(/\\?.*|/)$");
$details_capture = Pire::Capture("(/api/(v[3,4](/[\\w,\\-]+/|/)|frontend(/[\\w,\\-]+/|/)|/)[\\w,\\-]+/)");

UPSERT INTO {stat_cluster}.`{report_path}`
    SELECT  '{table_name}' as fielddate,
            http_method,
            request_path,
            PERCENTILE(execution_time, 0.90) as percentile_90,
            PERCENTILE(execution_time, 0.95) as percentile_95,
            PERCENTILE(execution_time, 0.99) as percentile_99,
            COUNT(*) as requests_count,
            COUNT_IF(status < 300) AS ok_responses_count,
            COUNT_IF(status > 499) AS bad_5xx_responses_count
       FROM (
            SELECT COALESCE($capture(request), "") as request_path,
                   CAST(process_time as Double) as execution_time,
                   COALESCE(method, "") as http_method,
                   status
            FROM `{log_path}{table_name}`
            WHERE domain = '{domain}'
               AND $patch_match(request)
               AND $capture(request) is NOT NULL
            UNION ALL

            SELECT COALESCE($details_capture(request) || "<id>", "") as request_path,
                IF(status < 300, CAST(process_time as Double)) as execution_time,
                COALESCE(method, "") as http_method,
                status
            FROM `{log_path}{table_name}`
            WHERE domain = '{domain}'
                AND $details_path_match(request)
       )
   GROUP BY http_method, request_path;
"""

STAT_CLUSTERS = {
    'testing': 'stat_beta',
    'production': 'stat',
}

MINUTE_LOG = '5_min'
DAY_LOG = '1_day'

LOG_LAG = {
    MINUTE_LOG: 30,  # 5-минутные логи отгружаем с задержкой в 30 минут (время в минутах)
    DAY_LOG: 24 * 60,  # Дневные логи отгружаем с задержкой в один день (время в минутах)
}

REPORT_PATH = {
    MINUTE_LOG: 'ExtData/ABC/api/api_response_time_day_5min/minutely',
    DAY_LOG: 'ExtData/ABC/api/api_response_time_day/daily',
}

LOG_PATH = {
    MINUTE_LOG: 'home/logfeller/logs/intranet-awacs-access-logs/stream/5min/',
    DAY_LOG: 'home/logfeller/logs/intranet-awacs-access-logs/1d/',
}

DATE_FORMAT = {
    MINUTE_LOG: '%Y-%m-%dT%H:%M:%S',
    DAY_LOG: '%Y-%m-%d',
}

DOMAIN = {
    'testing': 'abc-back.test.yandex-team.ru',
    'production': 'abc-back.yandex-team.ru'
}

moscow_tz = pytz.timezone('Europe/Moscow')


def get_table_name(log_type, report_date=None):
    if report_date is None:
        report_date = timezone.now() - timezone.timedelta(minutes=LOG_LAG[log_type])
    if log_type == MINUTE_LOG:
        truncated_minutes = report_date.minute - report_date.minute % 5
        report_date = report_date.replace(minute=truncated_minutes, second=0).astimezone(moscow_tz)
    return report_date.strftime(DATE_FORMAT[log_type])


def api_response(log_type):
    environment = yenv.type
    table_name = get_table_name(log_type=log_type)

    full_query = YQL_QUERY.format(
        table_name=table_name,
        domain=DOMAIN[environment],
        stat_cluster=STAT_CLUSTERS[environment],
        report_path=REPORT_PATH[log_type],
        log_path=LOG_PATH[log_type],
    )

    yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)
    request = yql_client.query(query=full_query, syntax_version=1)
    request.run()


@lock_task
def api_response_day():
    api_response(DAY_LOG)


@lock_task
def api_response_minute():
    api_response(MINUTE_LOG)
