from typing import List

from django.db.models import F
from django.utils import timezone

from metrics_framework.decorators import metric

from plan.services.models import ServiceCreateRequest, ServiceMoveRequest


def get_not_interactive_intervals() -> List[timezone.timedelta]:
    metric_scope_days = 7
    times = []
    for model in (ServiceCreateRequest, ServiceMoveRequest):
        ok_qs = (
            model.objects
            .filter(
                completed_at__isnull=False,
                interactive_at__isnull=False,
                interactive_at__gte=timezone.now() - timezone.timedelta(days=metric_scope_days),
            )
            .annotate(dt=F('interactive_at') - F('completed_at'))
        )

        not_ok_qs = (
            model.objects
            .filter(
                completed_at__isnull=False,
                interactive_at__isnull=True,
            )
            .annotate(dt=timezone.now() - F('completed_at'))
        )

        for qs in (ok_qs, not_ok_qs):
            times.extend(qs.values_list('dt', flat=True))

    return times


@metric('service_is_interactive_after')
def service_is_interactive_after():
    times = get_not_interactive_intervals()

    return [
        {
            'slug': 'count',
            'value': len(times),
        },
        {
            'slug': 'p99',
            'value': sorted(times)[int(len(times) * 0.99)].total_seconds() if times else 0,
        },
    ]
