from django.conf import settings
from django.db.models import F, Count

from metrics_framework.decorators import metric

from plan.services.models import Service, ServiceSuspiciousReason
from plan.suspicion.models import ServiceIssue, ServiceExecutionAction


@metric('other_services')
def count_other_services():
    meta_other = Service.objects.get(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)
    return [{
        'slug': 'other_services',
        'value': meta_other.get_descendants().filter(state__in=Service.states.ACTIVE_STATES).count()
    }]


@metric('service_statuses')
def count_services_by_statuses():
    return (
        Service.objects
        .values('state')
        .annotate(slug=F('state'))
        .annotate(value=Count('id'))
        .values('slug', 'value')
    )


@metric('suspicious_reasons')
def count_services_by_suspicious_reasons():
    return (
        ServiceSuspiciousReason.objects
        .filter(marked_by=None)
        .annotate(slug=F('reason'))
        .values('slug')
        .annotate(value=Count('service_id', distinct=True))
    )


@metric('suspicious_services')
def count_suspicious_services():
    return [
        {'slug': 'suspicious', 'value': Service.objects.filter(suspicious_date__isnull=False).count()},
        {'slug': 'not_suspicious', 'value': Service.objects.filter(suspicious_date__isnull=True).count()},
    ]


@metric('services_by_issues')
def calculate_services_by_issues():
    sandbox = Service.objects.get_sandbox()
    data = (
        ServiceIssue.objects
        .active()
        .filter(issue__isnull=False)
        .exclude(service__parent_id=sandbox.id)
        .values_list('issue__code')
        .annotate(count=Count('issue__code'))
        .values_list('issue__code', 'count')
    )
    result = []
    for issue, count in data:
        result.append({
            'context': {'issue': issue},
            'values': [{'slug': 'count', 'value': count}]
        })
    return result


@metric('services_by_issue_groups')
def calculate_services_by_issue_groups():
    sandbox = Service.objects.get_sandbox()
    data = (
        ServiceIssue.objects
        .active()
        .filter(issue_group__isnull=False)
        .exclude(service__parent_id=sandbox.id)
        .values_list('issue_group__code')
        .annotate(count=Count('issue_group__code'))
        .values_list('issue_group__code', 'count')
    )
    result = []
    for issue_group, count in data:
        result.append({
            'context': {'issue_group': issue_group},
            'values': [{'slug': 'count', 'value': count}]
        })
    return result


@metric('services_by_execution_steps')
def calculate_services_by_execution_steps():
    sandbox = Service.objects.get_sandbox()
    data = (
        ServiceExecutionAction.objects
        .future()
        .exclude(service_issue__service__parent_id=sandbox.id)
        .values_list('execution__code')
        .annotate(count=Count('execution__code'))
        .values_list('execution__code', 'count')
    )
    result = []
    for execution, count in data:
        result.append({
            'context': {'execution': execution},
            'values': [{'slug': 'count', 'value': count}]
        })
    return result
