from typing import List

from django.db.models import F, Q
from django.db.models.functions import Coalesce
from django.utils import timezone
from metrics_framework.decorators import metric

from plan.resources.constants import FOUND_IN_STAFF_TIME_LIMITS
from plan.services.models import ServiceMember


def get_found_in_staff_times(limit_timedelta) -> List[timezone.timedelta]:
    return list(
        ServiceMember.objects
        .filter(
            Q(found_in_staff_at__gt=timezone.now() - limit_timedelta) |
            Q(found_in_staff_at=None)
        )
        .annotate(timedelta=Coalesce(F('found_in_staff_at'), timezone.now()) - F('created_at'))
        .values_list('timedelta', flat=True)
    )


@metric('found_in_staff_after')
def found_in_staff_after():
    limits = FOUND_IN_STAFF_TIME_LIMITS.values()
    percentiles = [90, 95, 100]
    return [
        {
            'slug': f'worst{int(limit.total_seconds()//3600)}h{int(percentile)}p',
            'value': times[index].total_seconds() if times else 0
        }
        for limit in limits
        for percentile in percentiles

        for times in [sorted(get_found_in_staff_times(limit))]
        for index in [int((len(times) - 1) * percentile/100)]
    ]
