import abc

from django.conf import settings
from django.db.models import F
from django.http import HttpResponse
from django.utils import timezone
from django.views.generic import View

from django_celery_beat.models import PeriodicTask

from plan.unistat.models import TaskMetric
from plan.resources.models import ServiceResource, Resource
from plan.staff.models import Staff


class BaseMonitoringView(View):
    __metaclass__ = abc.ABCMeta

    ok_code = 200
    crit_code = 400
    warning_code = 412
    errors_delimiter = ' | '

    @abc.abstractmethod
    def monitoring(self):
        pass

    def get(self, request, *args, **kwargs):
        response_data = 'ok'
        status_code = 200
        errors, code = self.monitoring()
        if errors:
            status_code = code
            if isinstance(errors, str):
                errors = [errors]
            response_data = self.errors_delimiter.join(errors)

        return HttpResponse(
            response_data,
            status=status_code,
        )


class FailedTasksView(BaseMonitoringView):
    def monitoring(self):
        periodic_tasks = [task.task.split('.')[-1] for task in PeriodicTask.objects.filter(enabled=True)]
        tasks_metrics = {task.task_name: task for task in TaskMetric.objects.all()}
        failed_tasks = []
        now = timezone.now()
        code = self.ok_code
        for task in periodic_tasks:
            task_metric = tasks_metrics.get(task)
            allowed_delay = settings.ALLOWED_TASK_DELAY_TIME.get(task)
            if task_metric is None:
                failed_tasks.append(f'{task} absent in task metrics')
                if code == self.ok_code:
                    code = self.warning_code
            elif task_metric.use_for_monitoring:
                if allowed_delay is None:
                    failed_tasks.append(f'{task} absent in ALLOWED_TASK_DELAY_TIME')
                    if code == self.ok_code:
                        code = self.warning_code
                else:
                    delta_seconds = (now - task_metric.last_success_end).total_seconds()
                    if delta_seconds > allowed_delay:
                        failed_tasks.append(f'{task} last finished at {delta_seconds/3600:.2f} hours ago')
                        code = self.crit_code
        return failed_tasks, code


class GrantingResourcesView(BaseMonitoringView):
    def monitoring(self):
        now = timezone.now()
        granting_resources = ServiceResource.objects.filter(
            state__in=ServiceResource.GRANT_IN_PROCESS,
            approved_at__lte=now - F('type__max_granting_time'),
        ).exclude(type__code=settings.ROBOT_RESOURCE_TYPE_CODE)
        granting_resources_count = granting_resources.count()
        if granting_resources_count:
            value = 'Resources hanging after approving: %s' % granting_resources_count
        else:
            value = ''
        return value, self.crit_code


class MissingResourcesRobotsView(BaseMonitoringView):
    def monitoring(self):
        all_robots = set(
            Staff.objects.filter(
                is_robot=True,
                is_dismissed=False
            ).values_list('login', flat=True)
        )
        all_resources_robots = set(
            Resource.objects.filter(
                type__code=settings.ROBOT_RESOURCE_TYPE_CODE,
            ).values_list('external_id', flat=True)
        )

        code = self.ok_code
        value = ''

        missing_resources_robots = sorted(all_robots - all_resources_robots)
        if missing_resources_robots:
            code = self.crit_code
            value = 'Does not exist resources: %s. Logins robots: %s' % (
                len(missing_resources_robots), missing_resources_robots
            )

        return value, code
