from django.db import models

import django_filters
from plan.services import models as services_models
from plan.roles import models as roles_models
from plan.api.filters import PlanFilterSet, CustomModelChoiceFilter


class ServiceFilter(PlanFilterSet):
    possible_parent = django_filters.BooleanFilter(
        method='filter_possible_parent'
    )

    only_duty_responsible = django_filters.BooleanFilter(
        method='filter_duty_responsible'
    )

    service_type = CustomModelChoiceFilter(
        field_name='service_type',
        queryset=services_models.ServiceType.objects.all(),
        to_field_name='code',
        distinct=True,
    )

    class Meta:
        model = services_models.Service
        fields = ['possible_parent', 'service_type', 'only_duty_responsible']

    def filter_possible_parent(self, queryset, name, value):
        if value:
            return queryset.filter(
                level__lt=services_models.MAX_SERVICE_LEVEL-1,
                state__in=services_models.Service.states.ACTIVE_STATES,
            )
        return queryset

    def filter_duty_responsible(self, queryset, name, value):
        if value:
            return queryset.with_responsible(
                person=self.request.user,
                with_duty=True,
                with_descendants=True,
            )
        return queryset


class RoleFilter(PlanFilterSet):
    service = django_filters.CharFilter(
        method='filter_service', required=False
    )

    class Meta:
        model = roles_models.Role
        fields = ['service']

    def filter_service(self, queryset, name, value):
        if value:
            try:
                value = int(value)
            except ValueError:
                return queryset
            return queryset.filter(
                models.Q(service_id=value) | models.Q(service_id=None)
            )
        else:
            return queryset.filter(service_id=None)
