import waffle

from collections import defaultdict
from django.conf import settings
from django.core.urlresolvers import reverse
from django.db.models import (
    Q, Case, When, IntegerField,
    OuterRef, Exists,
)
from django.db.models.functions import Length
from multic.resources import SearchResource

from plan.internal_roles.models import InternalRole
from plan.multic.filters import ServiceFilter, RoleFilter
from plan.resources import models as resources_models
from plan.roles.models import Role, RoleScope
from plan.services.api.team import RoleSerializer
from plan.services.helpers import guess_default_service_role
from plan.services.models import (
    Service, ServiceTag, ServiceType,
    MAX_SERVICE_LEVEL, ServiceMember,
)
from plan.staff.models import Department, Staff
from plan.staff.utils import get_i_field, calculate_abc_ext


class PlanResource(SearchResource):

    def search(self, queryset):
        """Фильтрация по поисковой строке."""
        full_search_query = Q()
        for word in self.q.split():
            full_search_query &= self.search_word(word)

        if full_search_query:
            return queryset.filter(full_search_query)
        else:
            return queryset.all()


class ServiceResource(PlanResource):
    name = 'service'
    queryset = Service.objects.alive().annotate(
        state_order=Case(
            When(state=Service.states.IN_DEVELOP, then=1),
            When(state=Service.states.SUPPORTED, then=2),
            When(state=Service.states.NEEDINFO, then=3),
            When(state=Service.states.CLOSED, then=4),
            When(state=Service.states.DELETED, then=5),
            default=99,
            output_field=IntegerField()
        ),
        name_length=Length('name'),
    ).order_by('state_order', 'name_length', 'name', '-level')
    filter_cls = ServiceFilter

    required_fields = ('name', 'slug', 'url', 'state', 'is_base_non_leaf')
    search_conditions = {
        str: ('name__icontains', 'slug__icontains', 'name_en__icontains'),
        int: ('id',),
    }

    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['name']
        },
        'is_base_non_leaf': {
            'required_fields': ['id'],
        },
        'url': {
            'required_fields': ['id'],
        },
    }

    def apply_filters(self):
        queryset = self.queryset
        if self.filters:
            queryset = self.filter_cls(self.filters, queryset, request=self.request).qs

        from_service_id = self.request.GET.get('from_service_id')
        if (
            from_service_id
            and self.request.GET.get('check_type') == '1'
            and waffle.switch_is_active(settings.SWITCH_CHECK_ALLOWED_PARENT_TYPE)
        ):
            service = Service.objects.get(pk=from_service_id)
            available_types = ServiceType.available_parents.through.objects.filter(
                from_servicetype_id=service.service_type_id
            ).values_list('to_servicetype_id', flat=True)
            queryset = queryset.filter(
                service_type_id__in=available_types
            )
        return queryset

    def extra__text(self, obj):
        return str(obj['name'])

    def extra_is_base_non_leaf(self, obj):
        return Service.objects.get(pk=obj['id']).is_base_non_leaf()

    def extra_url(self, obj):
        return reverse('services:service', kwargs={'pk_or_slug': obj['id']})


class ParentServiceResource(ServiceResource):
    name = 'parent_service'
    queryset = Service.objects.filter(
        level__lt=MAX_SERVICE_LEVEL-1
    ).active().extra(
        select={'name_length': 'CHAR_LENGTH(services_service.name)'},
        order_by=['name_length', 'name', '-level']
    )


class RoleResource(PlanResource):
    name = 'role'
    queryset = Role.objects.all().order_by('scope__name', 'name')

    filter_cls = RoleFilter

    required_fields = ('scope', 'service_id', '_name')
    search_conditions = ('name__icontains', 'name_en__icontains')

    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['name', 'name_en', 'native_lang'],
        },
        'scope': {
            'required_fields': ['scope'],
        },
        '_name': {
            'required_fields': ['name'],
        },
    }

    def extra__text(self, obj):
        return get_i_field(obj, 'name')

    def extra_scope(self, obj):
        return {
            'id': obj['scope'],
            'name': RoleScope.objects.get(pk=obj['scope']).i_name,
        }

    def extra__name(self, obj):
        return {
            'ru': obj.get('name'),
            'en': obj.get('name_en')
        }


class StaffResource(PlanResource):
    name = 'staff'
    queryset = Staff.objects.filter(is_dismissed=False)

    required_fields = ('login', 'first_name', 'last_name',
                       'department__name', 'work_email', 'abc_ext')
    search_conditions = ('login__istartswith', 'first_name__istartswith',
                         'last_name__istartswith', 'first_name_en__istartswith',
                         'last_name_en__istartswith')
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['first_name', 'last_name',
                                'first_name_en', 'last_name_en'],
        },
        'first_name': {
            'required_fields': ['first_name', 'first_name_en', 'native_lang'],
        },
        'last_name': {
            'required_fields': ['last_name', 'last_name_en', 'native_lang'],
        },
        'department__name': {
            'required_fields': ['department__name', 'department__name_en',
                                'department__native_lang'],
        },
        'default_service_role': {
            'required_fields': ['id'],
        },
        'abc_ext': {
            'required_fields': ['id']
        },
    }

    def apply_filters(self):
        qs = super().apply_filters()
        try:
            service_ids = self.request.GET.get('from_service_id', '').split(',')
            from_service_ids = [int(service_id) for service_id in service_ids]
        except (TypeError, ValueError):
            from_service_ids = None

        if from_service_ids:
            is_member_of = ServiceMember.objects.filter(
                staff_id=OuterRef('pk'),
                service_id__in=from_service_ids,
                state=ServiceMember.states.ACTIVE,
            )

            qs = qs.annotate(
                is_member_of=Exists(is_member_of)
            ).filter(is_member_of=True)
        return qs

    def get_queryset_values(self, queryset):
        values = super(StaffResource, self).get_queryset_values(queryset)

        if 'default_service_role' in self.requested_extra:
            try:
                service_id = int(self.request.GET.get('_service_id'))
            except (TypeError, ValueError):
                service_id = None

            self.default_service_roles = guess_default_service_role(
                person_ids=[person['id'] for person in values],
                individual_results=True,
                service_id=service_id,
            )

        return values

    def extra__text(self, obj):
        return ' '.join([self.extra_first_name(obj), self.extra_last_name(obj)])

    def extra_first_name(self, obj):
        return get_i_field(obj, 'first_name')

    def extra_last_name(self, obj):
        return get_i_field(obj, 'last_name')

    def extra_department__name(self, obj):
        return get_i_field(obj, 'name', 'department__')

    def extra_default_service_role(self, obj):
        return RoleSerializer(self.default_service_roles[obj['id']]).data

    def extra_abc_ext(self, obj):
        return calculate_abc_ext(obj['login'])


class DepartmentResource(PlanResource):
    name = 'department'
    queryset = Department.objects.active()

    required_fields = ('id', 'name', 'url', 'abc_ext')
    search_conditions = ('name__icontains', 'name_en__icontains', 'url__istartswith')
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
            },
        '_text': {
            'required_fields': ['name'],
            },
        'name': {
            'required_fields': ['name', 'name_en', 'native_lang'],
            },
        'abc_ext': {
            'required_fields': ['id']
        },
    }

    def extra__text(self, obj):
        return str(self.extra_name(obj))

    def extra_name(self, obj):
        return get_i_field(obj, 'name')

    def extra_abc_ext(self, obj):
        departament = Department.objects.get(pk=obj['id'])
        departments = departament.get_descendants(include_self=True).values_list('id', flat=True)

        all_staff = set(
            Staff.objects
            .filter(is_dismissed=False, department__id__in=departments)
            .values_list('id', flat=True)
        )

        codes_roles = [settings.FULL_ACCESS, settings.SERVICES_VIEWER, settings.OWN_ONLY_VIEWER]
        all_access = InternalRole.objects.filter(
            staff__in=all_staff,
            role__in=codes_roles
        ).values_list('role', 'staff_id')

        access = defaultdict(set)
        for role, staff_id in all_access:
            access[role].add(staff_id)

        abc_ext = []
        staffs_with_full_access = set()
        staffs_with_services_viewer = set()

        if settings.FULL_ACCESS in access and access[settings.FULL_ACCESS]:
            staffs_with_full_access = access[settings.FULL_ACCESS]

        if staffs_with_full_access == all_staff:
            return abc_ext

        if settings.SERVICES_VIEWER in access and access[settings.SERVICES_VIEWER]:
            staffs_with_services_viewer = access[settings.SERVICES_VIEWER]
            if staffs_with_full_access:
                staffs_with_services_viewer -= staffs_with_full_access

            if staffs_with_services_viewer:
                abc_ext.append(settings.HAS_SERVICES_VIEWER)
                if all_staff == staffs_with_full_access | staffs_with_services_viewer:
                    return abc_ext

        if settings.OWN_ONLY_VIEWER in access and access[settings.OWN_ONLY_VIEWER]:
            staffs_with_own_only_viewer = access[settings.OWN_ONLY_VIEWER]
            if staffs_with_full_access:
                staffs_with_own_only_viewer -= staffs_with_full_access
            if staffs_with_services_viewer:
                staffs_with_own_only_viewer -= staffs_with_services_viewer

            if staffs_with_own_only_viewer:
                abc_ext.append(settings.HAS_OWN_ONLY_VIEWER)

        return abc_ext


class ResourceFormTypeResource(PlanResource):
    name = 'resource_form_type'
    queryset = (
        resources_models.ResourceType.objects
        .filter(Q(is_enabled=True) & (Q(form_id__isnull=False) | Q(has_multiple_consumers=True)))
        .order_by('name')
    )
    filter_fields = ('supplier', 'category')

    required_fields = ('name', 'form_id', 'supplier__name')
    search_conditions = ('name__icontains', 'supplier__name__icontains',)
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['name'],
        },
    }

    def extra__text(self, obj):
        return str(obj['name'])


class ResourceTagResource(PlanResource):
    name = 'resource_tag'
    queryset = resources_models.ResourceTag.objects.all().select_related('category').order_by('category__order', 'name')
    filter_fields = ('name', 'service', 'resource_type')

    required_fields = ('name', 'slug', 'category__name', 'service_id')
    search_conditions = ('name__icontains', 'name_en__icontains', 'slug__icontains',)
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['name'],
        },
        'category__name': {
            'required_fields': ['category__name'],
        },
    }

    def apply_filters(self):
        if self.filters:
            service = self.filters.pop('service', None)
            resource_type = self.filters.pop('resource_type', None)

            queryset = super(ResourceTagResource, self).apply_filters()

            if service:
                queryset = queryset.filter(Q(service_id=service) | Q(service=None))

            if resource_type:
                resource_type = resources_models.ResourceType.objects.get(id=resource_type)
                if queryset.filter(type_tags=resource_type).exists():
                    # Теги, привязанные к этому типу ресурса
                    queryset = queryset.filter(type_tags=resource_type)

                else:
                    # Теги, привязанные к поставщику этого типа или глобальные теги
                    queryset = queryset.filter(Q(service=resource_type.supplier) | Q(service=None))

            return queryset
        else:
            return super(ResourceTagResource, self).apply_filters()

    def extra__text(self, obj):
        return obj['name']


class ResourceResource(PlanResource):
    name = 'resource'
    queryset = resources_models.Resource.objects.all().order_by('name')
    filter_fields = ('type',)

    required_fields = ('external_id', 'name')
    search_conditions = ('name__icontains', 'external_id__icontains',)
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['name'],
        },
    }

    def extra__text(self, obj):
        return str(obj['name'])


class ServiceResourceResource(PlanResource):
    name = 'service_resource'
    queryset = resources_models.ServiceResource.objects.filter(state='granted').order_by('resource__name')
    filter_fields = ('service',)

    required_fields = ('external_id', 'resource_id')
    search_conditions = ('resource__name__icontains', 'resource__external_id__icontains',)
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['resource'],
        },
        'external_id': {
            'required_fields': ['resource'],
        },
        'resource_id': {
            'required_fields': ['resource'],
        },
    }

    def extra_resource_id(self, obj):
        return obj['resource']

    def extra__text(self, obj):
        resource = resources_models.Resource.objects.get(pk=obj['resource'])
        return str(resource.name)

    def extra_external_id(self, obj):
        resource = resources_models.Resource.objects.get(pk=obj['resource'])
        return str(resource.external_id)


class ServiceTagResource(PlanResource):
    name = 'service_tag'
    queryset = ServiceTag.objects.order_by('name')

    required_fields = ('id', 'name', 'name_en', 'color', 'slug')
    search_conditions = ('name__icontains', 'name_en__icontains',)
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['name'],
        },
    }

    def extra__text(self, obj):
        return obj['name']


class ResourceTypeCategoryResource(PlanResource):
    name = 'type_category'
    queryset = resources_models.ResourceTypeCategory.objects.order_by('name')
    filter_fields = ('name',)

    required_fields = ('id', 'name', 'name_en', 'slug')
    search_conditions = ('name__icontains', 'name_en__icontains', 'slug__icontains')
    extra_fields = {
        '_id': {
            'required_fields': ['id'],
        },
        '_text': {
            'required_fields': ['name'],
        },
    }

    def apply_filters(self):
        queryset = super(ResourceTypeCategoryResource, self).apply_filters()

        if self.filters:
            with_types = self.filters.pop('with_types', None)
            if with_types:
                queryset = queryset.filter(resourcetype__is_enabled=True).distinct()

        return queryset

    def extra__text(self, obj):
        return obj['name']
