import functools
import itertools

from plan.notify.core.context import ContextAggregator
from plan.services.models import ServiceMemberDepartment

from . import generators


class ServiceTeamAggregator(ContextAggregator):
    def __init__(self, *args, **kwargs):
        self._grouping_map = kwargs.pop('grouping_map', None) or {'*': None}
        super(ServiceTeamAggregator, self).__init__(*args, **kwargs)

    @staticmethod
    def format_member(member):
        return '{}:{}'.format(member['person'].id, member['role'].id)

    def group_by(self, field, value):
        if isinstance(value, list):
            result = []
            for sender, items in itertools.groupby(value, lambda i: i[field]):
                result.append({field: sender, 'items': list(items)})
            return result
        else:
            return value

    def maybe_group(self, key, items):
        default_field = self._grouping_map.get('*')
        field = self._grouping_map.get(key, default_field)

        if field:
            return self.group_by(field, items)
        return items

    def preprocess_context(self, context):
        return context

    def __iter__(self):
        pself = super(ServiceTeamAggregator, self)
        for (recipient, service), context in pself.__iter__():
            context['service'] = service
            context = self.preprocess_context(context)

            yield recipient, {k: self.maybe_group(k, v)
                              for k, v in context.items()}


class ServiceTeamApproveAggregator(ServiceTeamAggregator):
    """
    Этот агрегатор группирует события по автору и выделяет некоторые из них
    в отдельный список members_approve_edited
    """

    def preprocess_context(self, context):
        context['members_approve_edited'] = []
        ids = []
        for key, value in context.items():
            if key == 'members_approve':
                newvalue = []
                for item in value:
                    if item.get('edited', False):
                        context['members_approve_edited'].append(item)

                        ids.append(self.format_member({
                            'person': item['person'],
                            'role': item['new']['role']
                        }))
                    else:
                        newvalue.append(item)
                        ids.append(self.format_member(item))
                context[key] = newvalue

        context['querystring_ids_digest'] = 'approve_members=%s' % ','.join(ids)

        return context


class ServiceApproveDepartmentAggregator(ServiceTeamAggregator):
    def preprocess_context(self, context):
        ids = []
        for item in context['departments']:
            ids.append(str(item['department'].id))

        context['querystring_ids_digest'] = \
            'approve_departments=%s' % ','.join(ids)

        return context


class ServiceApproveDepartmentRemovalAggregator(ServiceTeamAggregator):
    def preprocess_context(self, context):
        ids = []
        for item in context['departments']:
            ids.append(str(item['department'].id))

        context['querystring_ids_digest'] \
            = 'remove_departments=%s' % ','.join(ids)

        return context


class ServiceApproveReminderAggregator(ServiceTeamAggregator):
    def preprocess_context(self, context):
        query_params = []

        if 'members' in context:
            ids = []
            for member in context['members']:
                ids.append(self.format_member(member))
            query_params.append('approve_members=%s' % ','.join(ids))

        if 'departments_approve' in context:
            ids = []
            for item in context['departments_approve']:
                ids.append(str(item['department'].id))
            query_params.append('approve_departments=%s' % ','.join(ids))

        if 'departments_remove' in context:
            ids = []
            for item in context['departments_remove']:
                ids.append(str(item['department'].id))
            query_params.append('remove_departments=%s' % ','.join(ids))

        context['querystring_ids_digest'] = '&'.join(query_params)

        return context


# рассылка "изменение команды сервиса" владельцу и главному в роли
team_changed = functools.partial(
    ServiceTeamAggregator,
    generators=(
        generators.PersonRemovedGenerator,
        generators.PersonAddedGenerator,
    ),
    ensure_uniqueness=generators.PersonAddedGenerator.ensure_uniqueness,
    grouping_map={
        'members_removed': 'sender',
        '*': 'from_department',
    }
)

# рассылка "вы удалены из команды сервиса"
you_were_removed = functools.partial(
    ServiceTeamAggregator,
    generators=(
        generators.YouWereRemovedGenerator,
    ),
    ensure_uniqueness=generators.YouWereRemovedGenerator.ensure_uniqueness,
)

# рассылка "вы добавлены в команду сервиса"
you_were_added = functools.partial(
    ServiceTeamAggregator,
    generators=(
        generators.YouWereAddedGenerator,
    ),
    ensure_uniqueness=generators.YouWereAddedGenerator.ensure_uniqueness,
)


# еженедельное напоминание про команду сервисов
weekly_reminder = functools.partial(
    ServiceApproveReminderAggregator,
    data=itertools.chain(
        ServiceMemberDepartment.objects.all(),
    ),
    generators=[],
)
