import logging

from plan.common.utils.oauth import get_abc_zombik
from plan.notify.core.generators import NotificationGenerator
from plan.notify.recipients import ToolsRecipient
from plan.roles.models import Role
from plan.services.models import Service, ServiceMember
from plan.staff.models import Department, Staff

log = logging.getLogger(__name__)


class EventBaseGenerator(NotificationGenerator):
    event_type = None

    unique_fields = ()

    @classmethod
    def trim_non_unique_fields(cls, fragment):
        return {k: v for k, v in fragment.items() if k in cls.unique_fields}

    def is_entry_relevant(self, entry):
        return entry['type'] == self.event_type

    def get_excluded_recipients(self, entry):
        return {(self._get_sender(entry), self._get_service(entry))}

    def _get_sender(self, entry):
        if not hasattr(self, '_sender_cache'):
            self._sender_cache = Staff.objects.get(pk=entry['sender_id'])
        return self._sender_cache

    def _get_person(self, entry):
        if not hasattr(self, '_person_cache'):
            self._person_cache = Staff.objects.get(
                pk=entry['params']['person_id'],
            )
        return self._person_cache

    def _get_role(self, entry):
        if not hasattr(self, '_role_cache'):
            self._role_cache = Role.objects.get(
                pk=entry['params']['role_id'],
            )
        return self._role_cache

    def _get_service(self, entry):
        if not hasattr(self, '_service_cache'):
            self._service_cache = Service.objects.select_related('owner').get(
                pk=entry['params']['service_id'],
            )
        return self._service_cache

    def _get_department(self, entry):
        if not hasattr(self, '_department_cache'):
            self._department_cache = Department.objects.get(
                pk=entry['params']['from_department_id'],
            )
        return self._department_cache

    @classmethod
    def ensure_uniqueness(cls, fragment, fragments):
        trim = cls.trim_non_unique_fields
        return trim(fragment[1]) not in [trim(f) for f in fragments]


class PersonBaseGenerator(EventBaseGenerator):
    unique_fields = ('person', 'role')

    def get_fragment_value(self, entry, recipient):
        value = {
            'person': self._get_person(entry),
            'role': self._get_role(entry),
            'sender': self._get_sender(entry),
        }

        if entry['params'].get('from_department_id'):
            value['from_department'] = self._get_department(entry)

        return value

    def get_recipients(self, entry):
        recipients = set()

        service = self._get_service(entry)

        if service.owner:
            recipients.add((service.owner, service))

        for responsible in ServiceMember.objects.responsibles():
            recipients.add((responsible.staff, service))

        return recipients - self.get_excluded_recipients(entry)

    def get_base_queryset(self, entry):
        try:
            return ServiceMember.objects.filter(
                service__id=entry['params']['service_id'],
                staff__id=entry['params']['person_id'],
                role=self._get_role(entry),
            )

        except Role.DoesNotExist:
            # некорректные данные в логе
            log.warning('Bad role_id in entry %r, class %s', entry, self.__class__.__name__)
            return ServiceMember.objects.none()


class PersonRemovedGenerator(PersonBaseGenerator):
    event_type = 'service_member_removed'

    unique_fields = ('person', 'role', 'from_department')

    def get_context_key(self, entry):
        if entry['params']['from_department_id']:
            return 'members_auto_removed'
        else:
            return 'members_removed'


class PersonAddedGenerator(PersonBaseGenerator):
    event_type = 'service_member_added'
    context_key = 'members_auto_added'

    unique_fields = ('person', 'role', 'from_department')

    def is_entry_relevant(self, entry):
        pself = super(PersonAddedGenerator, self)
        if not pself.is_entry_relevant(entry):
            return False

        if not entry['params'].get('from_department_id'):
            return False

        queryset = self.get_base_queryset(entry).exclude(from_department=None)
        return queryset.exists()


class YouWereRemovedGenerator(PersonBaseGenerator):
    event_type = 'service_member_removed'

    unique_fields = ()

    def get_recipients(self, entry):
        recipients = {(self._get_person(entry), self._get_service(entry))}
        return recipients - self.get_excluded_recipients(entry)

    def get_context_key(self, entry):
        if entry['params']['from_department_id']:
            return 'auto_removed_from'
        else:
            return 'removed_from'

    def get_fragment_value(self, entry, recipient):
        value = {
            'sender': self._get_sender(entry),
        }

        if entry['params']['from_department_id']:
            value['from_department'] = self._get_department(entry)

        return value


class YouWereAddedGenerator(PersonBaseGenerator):
    event_type = 'service_member_added'
    unique_fields = ()

    def get_recipients(self, entry):
        recipients = {(self._get_person(entry), self._get_service(entry))}
        return recipients - self.get_excluded_recipients(entry)

    def get_context_key(self, entry):
        planner_user = get_abc_zombik()
        sender = self._get_sender(entry)
        if entry['params']['from_department_id'] and sender == planner_user:
            return 'auto_added_to'
        else:
            return 'added_to'


class DepartmentBaseGenerator(EventBaseGenerator):
    context_key = 'departments'
    unique_fields = ('department',)

    def _get_department(self, entry):
        if not hasattr(self, '_department_cache'):
            self._department_cache = Department.objects.get(
                pk=entry['params']['department_id'],
            )
        return self._department_cache

    def get_fragment_value(self, entry, recipient):
        return {
            'department': self._get_department(entry),
            'sender': self._get_sender(entry),
        }

    def get_recipients(self, entry):
        service = self._get_service(entry)
        recipients = set()
        if service.owner:
            recipients.add((service.owner, service))
        return recipients - self.get_excluded_recipients(entry)


class QuerysetBaseGenerator(NotificationGenerator):
    def get_recipients(self, entry):
        return {(entry.service.owner or ToolsRecipient(), entry.service)}


class QuerysetDepartmentBase(QuerysetBaseGenerator):
    def get_fragment_value(self, entry, recipient):
        return {
            'department': entry.department,
        }
