from collections import defaultdict


class ContextAggregator(object):
    """
    Реализует обработку полученных данных для рассылки нотификаций:
      * вытаскивание из данных получателей
      * формирование контекста для каждого получателя
    """
    def __init__(self, data, generators, **params):
        """
        :param data: Данные в виде последовательности или нескольких
            последовательностей. Если данные требуют предобработки,
            рекомендуется наследовать этот класс и переопределять build_entries.
        :param generators: Последовательность генераторов --
            наследников NotificationGenerator, в которых содержится логика по
            формированию списка получателей кусочков контекста для получателя.
        :param params:
          * ensure_uniqueness -- если True или функция, то при группировке
            кусочков фрагментов по ключу будут выкидываться неуникальные кусочки.
        """
        self.data = data
        self.generators = generators
        self.ensure_uniqueness = params.pop('ensure_uniqueness', False)
        super(ContextAggregator, self).__init__(**params)

    def __iter__(self):
        """
        Основной метод, по инстансу класса нужно итерироваться.
        :returns Каждая итерация возвращает пару (получатель, контекст)
        """
        for recipient, fragments in self.get_fragments():
            yield recipient, self.gather_context(fragments)

    def get_fragments(self):
        """
        Предобработанные данные передаются во ВСЕ генераторы из self.generators.
        :returns пары (получатель, список фрагментов),
            где фрагмент == (ключ для контекста, значение)
        """
        results = defaultdict(list)
        for entry in self.build_entries(self.data):
            for generator in self.generators:
                for recipient, fragments in generator(entry):
                    results[recipient].extend(fragments)

        return iter(results.items())

    def build_entries(self, data):
        """
        Хук для предобработки данных перед передачей в генераторы.
        Если в этом месте много логики, ее можно выделить в
        наследнике класса NotificationGeneratorBackend. Но это опционально,
        если тут нужно написать пару строчек, то можно просто написать их в
        build_entries и не усложнять цепочку.
        """
        return data

    def gather_context(self, fragments):
        """
        Сгруппировать список фрагментов по ключу.
        """
        context = {}
        for fragment in fragments:
            self.add_fragment_to_context(context, fragment)
        return context

    def add_fragment_to_context(self, context, fragment):
        key, value = fragment
        fragments = context.setdefault(key, [])

        if callable(self.ensure_uniqueness):
            if not self.ensure_uniqueness(fragment, fragments):
                return
        elif self.ensure_uniqueness and value in fragments:
            return

        fragments.append(value)
