import logging

import waffle
import yenv
from django.conf import settings
from django.core.mail import EmailMessage

from plan.common.utils.tasks import lock_task, retry_exponential
from plan.staff.models import Staff

logger = logging.getLogger(__name__)


def send_mail_basic(
    subject,
    body,
    from_email=None,
    to=None,
    content_subtype='html',
    cc=None,
    bcc=None,
    reply_to=None,
    attach=None,
):

    if yenv.type not in ('production', 'development.unittest'):
        body += "<br/>\nOriginal To: %s, CC: %s, BCC: %s" % (to, cc, bcc)
        to = [settings.DEBUG_EMAIL]
        cc = None
        bcc = None
    else:
        cc = set(cc) if cc else set()
        for email in settings.DEFAULT_CC_EMAILS:
            cc.add(email)
        cc = list(cc)

    headers = {}
    if reply_to:
        if isinstance(reply_to, Staff):
            headers['Reply-To'] = reply_to.get_email()
        else:
            headers['Reply-To'] = reply_to

    msg = EmailMessage(
        subject=subject,
        body=body,
        from_email=from_email or settings.DEFAULT_FROM_EMAIL,
        to=to,
        cc=cc,
        bcc=bcc,
        headers=headers,
    )
    msg.content_subtype = content_subtype

    if attach:
        msg.attach(*attach)

    if not settings.SEND_EMAIL or waffle.switch_is_active('dont_send_email'):
        logger.info(
            'Fake email %(subj)s to %(to)s',
            {'subj': subject, 'to': to}
        )
        return

    msg.send()
    logger.info('Sent email %(subj)s to %(to)s', {'subj': subject, 'to': to})


send_mail = lock_task(lock=False, rate_limit="100/m")(send_mail_basic)
send_mail = retry_exponential(send_mail)
