import logging
from itertools import chain

logger = logging.getLogger(__name__)


class NotificationGenerator(object):
    """
    Абстрактный базовый класс, нужно наследовать.
    Содержит в себе логику по формированию списка получателей и кусочков
    контекста.
    Получает данные из ContextAggregator, которые могут быть, а могут и не быть
    там предварительно обработаны для генератора.
    """
    context_key = None
    recipient_getters = ()

    def __init__(self, entry):
        super(NotificationGenerator, self).__init__()
        self.entry = entry

    def __iter__(self):
        if not self.is_entry_relevant(self.entry):
            return

        for recipient in self.get_recipients(self.entry):
            fragments = self.extract_fragments(self.entry, recipient)
            if fragments:
                yield recipient, fragments

    def is_entry_relevant(self, entry):
        """
        Хук для переопределения в подклассах, если нужно, чтобы генератор
        получил лишь свою часть данных.
        """
        return True

    def get_recipients(self, entry):
        instance = self.get_instance(entry)

        recipients_lists = []
        for getter in self.recipient_getters:
            recipients_lists.append(list(filter(bool, getter(instance))))
        all_recipients = chain(*recipients_lists)

        exclude_recipients = self.get_excluded_recipients(entry)
        return set(all_recipients) - exclude_recipients

    def get_instance(self, entry):
        """
        :return: instance -- бизнес-сущность, которую можно скормить в
        соответствующий getter.
        """
        return entry

    def get_excluded_recipients(self, entry):
        return set()

    def extract_fragment(self, entry, recipient):
        """
        Можно переопределить метод, если нужно возвращать один фрагмент.
        Этот метод может возвращать фрагменты с пустым value или значения,
        которые сводятся к False, они будут отфильтрованы позже.
        """
        key = self.get_context_key(entry)
        value = self.get_fragment_value(entry, recipient)
        return key, value

    def extract_fragments(self, entry, recipient):
        """
        Можно переопределить метод, если нужно возвращать несколько фрагментов.
        Этот метод может возвратить пустой список, однако в нем не должно быть
        пустых фрагментов.
        """
        fragment = self.extract_fragment(entry, recipient)

        if fragment:
            key, value = fragment
            if value:
                return [fragment]

        return []

    def get_context_key(self, entry):
        """
        Этот метод можно переопределить, чтобы положить данные в контекст с
        ключом, отличным от self.context_key.
        """
        return self.context_key

    def get_fragment_value(self, entry, recipient):
        """
        Этот метод можно переопределить, чтобы описать сборку значения для
        фрагмента на основе entry для получателя recipient
        """
        return entry
