import logging

from django.template import loader, Context
from django.utils import translation


logger = logging.getLogger(__name__)


class Notification(object):

    def __init__(self, id):
        self.id = id

    def __str__(self):
        return '<Notification: {self.id}>'.format(self=self)

    __repr__ = __str__

    def render(self, context, format, lang):
        """
        Возвращает отрендеренный шаблон.
        format - 'html'|'txt'
        lang может быть как одним языком, так и списком языков в порядке
        предпочтения.
        """
        from django.template.exceptions import TemplateDoesNotExist

        try:
            template = self.load_template(format, lang)
        except TemplateDoesNotExist:
            return None

        logger.debug(str(self))
        logger.debug('Template: %s', template.name)

        found_lang = self.get_template_lang(template.name)
        with translation.override(found_lang):
            result = template.render(Context(context))

        return result

    def load_template(self, format, lang):

        if not isinstance(lang, (list, tuple)):
            langs = [lang]
        else:
            langs = lang

        templates_list = [
            self.base_template_name + '_' + lang_name + '.' + format
            for lang_name in langs
        ]
        return loader.select_template(templates_list).template

    @staticmethod
    def get_template_lang(template_name):
        name, extention = template_name.rsplit('.', 1)
        name, lang = name.rsplit('_', 1)
        return lang

    @property
    def base_template_name(self):
        return self.id.replace('.', '/')
