import six

from django.conf import settings

from plan.notify.core.email import send_mail


WHITESPACES = '\n\r\t '


class EmailTransport(object):

    def deliver(self, notification, context, recipients, **params):
        """
        Основной публичный метод, который делает всю работу, вызывая
        перегруженные методы подклассов.
        """
        composed = self.compose(notification, context, recipients)
        self.process(composed, recipients, **params)

    @staticmethod
    def get_language(recipient):
        language = 'ru'
        if not isinstance(recipient, six.string_types):
            language = recipient.lang_ui
        return language

    def compose(self, notification, context, recipients):
        """
        Письмо пока рендерим на каком-то одном языке. С фолбеком на русском.
        Если нужно более умную логику деградации языков, то можно ее тут
        внедрить позже.
        """
        preferred_langs = ['ru']
        if len(recipients) == 1:
            only_recipient_lang = self.get_language(recipients[0])
            if only_recipient_lang and only_recipient_lang != 'ru':
                preferred_langs.insert(0, only_recipient_lang)

        rendered = notification.render(
            context=context,
            format='html',
            lang=preferred_langs
        )

        return self.parse_rendered(rendered)

    @staticmethod
    def parse_rendered(rendered):
        """
        Разбить отрендеренный шаблон на заголовок и тело.
        Завернуть в объект, пригодный для process.
        """
        rendered = rendered.replace('\r', '')

        TPL_DELIMITER = '<-- SECTION -->'
        TPL_SECTIONS_NUMBER = 1
        header, body = rendered.split(TPL_DELIMITER, TPL_SECTIONS_NUMBER)
        header = header.strip(WHITESPACES)
        body = body.strip(WHITESPACES)

        return {
            'header': header,
            'body': body
        }

    def process(self, composed, recipients, **params):
        cc = params.pop('cc', None)
        bcc = params.pop('bcc', None)
        cc = cc and [self.fetch_email(c) for c in cc]
        bcc = bcc and [self.fetch_email(bc) for bc in bcc]

        sender = send_mail
        if settings.NOTIFY_EMAIL_ASYNC:
            sender = sender.delay

        sender(
            subject=composed['header'],
            body=composed['body'],
            to=[self.fetch_email(recipient) for recipient in recipients],
            cc=cc,
            bcc=bcc,
            **params
        )

    @staticmethod
    def fetch_email(person_or_email):
        email = person_or_email
        if not isinstance(person_or_email, six.string_types):
            email = person_or_email.email
        return email
