import logging


logger = logging.getLogger(__name__)


def normalize_recipients_lists(recipients):
    """
    Представляем получателей в общем виде: каждый элемент — это
    список из одного или более получателей.
    """
    list_of_lists = []
    for recipient_or_group in recipients:
        if isinstance(recipient_or_group, (list, tuple, set)):
            list_of_lists.append(recipient_or_group)
        else:
            list_of_lists.append([recipient_or_group])
    return list_of_lists


def filter_appropriate_recipients(recipients):
    """
    Составляем списки без пустых получателей и уволенных.
    А также выкидываем повторных (recipient должен быть hashable).
    """
    def is_appropriate(recipient):
        if not recipient:
            msg = 'Discard delivery for None because None is None'
            logger.info(msg)
            return False
        if hasattr(recipient, 'is_dismissed') and recipient.is_dismissed:
            msg = 'Discard delivery for %(login)s because of dismissal'
            logger.info(msg, {'login': recipient.login})
            return False
        return True

    clean_recipients = []

    for recipient_group in recipients:
        recipient_group = set(recipient_group)
        recipient_group = list(filter(is_appropriate, recipient_group))
        if recipient_group:
            clean_recipients.append(recipient_group)

    return clean_recipients


def build_extended_context(context, recipient_group):
    """
    Добавляем в контекст для нотификации информацию о получателях,
    чтобы использовать это в шаблоне.
    """
    updated_context = dict(context)
    if len(recipient_group) == 1:
        updated_context['recipient'] = recipient_group[0]
    else:
        updated_context['recipients'] = recipient_group
    return updated_context
