import collections
import logging

from django.conf import settings

from plan.common.utils.context import common_request_free_context
from plan.notify.core import transports
from plan.notify.core import utils
from plan.notify.core.notifications import Notification

logger = logging.getLogger(__name__)


def deliver_email(
        notification_id,
        recipients,
        context=None,
        **params):
    """
    notification_id - идентификатор нотификации
    recipients - list or list of lists получателей типа Staff
    params — передаются в транспорт без изменений
    """
    context = context or {}
    context.update(common_request_free_context())

    notification = Notification(id=notification_id)

    transport = transports.EmailTransport()

    recipients = utils.normalize_recipients_lists(recipients)
    recipients = utils.filter_appropriate_recipients(recipients)

    for recipient_group in recipients:
        updated_context = utils.build_extended_context(context, recipient_group)
        logger.info(
            'Ready to deliver notification %(notification)s '
            'to %(num)s recipients: %(recipients)s',
            {
                'notification': notification,
                'num': len(recipient_group),
                'recipients': ', '.join(r.login if hasattr(r, 'login') else str(r) for r in recipients),
            }
        )

        transport.deliver(
            notification=notification,
            context=updated_context,
            recipients=recipient_group,
            **params
        )


def send_to_team(notification_id, context):
    StaffTuple = collections.namedtuple('Staff', ('email', 'lang_ui'))
    recipient = StaffTuple(
        email=settings.ABC_TEAM_EMAIL,
        lang_ui='ru',
    )
    deliver_email(
        notification_id=notification_id,
        context=context,
        # здесь важно передавать получателя вложенным в два списка, т.к.
        # namedtuple сам по себе дальше развернется при обработке письма
        recipients=[[recipient]],
    )


__all__ = ['deliver_email', 'send_to_team']
